# coding: utf-8

from __future__ import unicode_literals

from datetime import date, datetime, timedelta
import logging
import urlparse

from django.conf import settings

from blackbox import Blackbox
from ids.registry import registry
from ticket_parser2.api.v1 import BlackboxClientId

from uhura.external import intranet
from uhura.external import staff
from uhura.lib import cache_storage
from uhura import models
from uhura.utils import datetimes, tvm

holidays = registry.get_repository(
    'calendar', 'holidays', user_agent='uhura', oauth_token=settings.ROBOT_TOKEN, retries=3
)
logger = logging.getLogger(__name__)


def cast_timestamps_to_readable_format(events):
    if events:
        for i in range(len(events)):
            events[i]['startTs'] = datetimes.datetime_str_to_readable_format(events[i]['startTs'])
            events[i]['endTs'] = datetimes.datetime_str_to_readable_format(events[i]['endTs'])


def request_to_calendar(method, params, credentials):
    url = urlparse.urljoin(settings.CALENDAR_API_URL, method)
    tvm_headers = tvm.get_tvm_service_headers(settings.TVM2_CALENDAR_ID)

    tvm_headers.update(tvm.get_tvm_user_headers(credentials.token))

    return intranet.get_request(url, params=params, headers=tvm_headers)


def post_request_to_calendar(method, params, data, credentials):
    url = urlparse.urljoin(settings.CALENDAR_API_URL, method)
    tvm_headers = tvm.get_tvm_service_headers(settings.TVM2_CALENDAR_ID)

    tvm_headers.update(tvm.get_tvm_user_headers(credentials))

    return intranet.post_request(url, params=params, data=data, headers=tvm_headers)


@cache_storage.memoize_decorator(fresh_time=7 * 24 * 3600, cache_none=False)
def get_holidays():
    return [x['date'] for x in holidays.get(lookup={
        'start_date': date.today(),
        'end_date': date.today() + timedelta(days=7),
        'who_am_i': 'uhura'
    })]


def is_working_day(day):
    if isinstance(day, datetime):
        day = day.date()
    return day not in get_holidays()


def prev_working_day(day):
    day -= timedelta(days=1)
    while not is_working_day(day):
        day -= timedelta(days=1)
    return day


class CalendarRequestCredentials(object):
    @staticmethod
    def uhura_credentials():
        return CalendarRequestCredentials({
            'uid': settings.ROBOT_UID,
            'login': settings.ROBOT_LOGIN,
        })

    def __init__(self, person_data):
        self._person_uid = person_data['uid']
        self._uid = None
        self._token = None
        self._user_has_token = False
        self.person_login = person_data['login']

    def _fallback_to_uhura_token(self):
        self._user_has_token = False
        self._token = settings.ROBOT_TOKEN
        self._uid = settings.ROBOT_UID

    def _update(self):
        if self._token:
            return

        token = models.UserToken.objects.filter(user_id=self._person_uid).first()

        if not token:
            self._fallback_to_uhura_token()
            return

        # We have to check user token on each call
        bb_answer = Blackbox().oauth(
            tvm.get_tvm_service_headers(BlackboxClientId.ProdYateam.value),
            userip='::1',
            oauth_token=token.token,
        )

        if bb_answer.get('status') != 'VALID':
            self._fallback_to_uhura_token()
            return

        self._user_has_token = True
        self._token = token.token
        self._uid = self._person_uid

    @property
    def user_has_token(self):
        self._update()
        return self._user_has_token

    @property
    def token(self):
        self._update()
        return self._token

    @property
    def uid(self):
        self._update()
        return self._uid


class MeetingsRequest(object):
    def __init__(self, subject_uid, from_date, to_date, credentials):
        # type: (str, datetime, datetime, CalendarRequestCredentials) -> None
        self.subject_uid = subject_uid

        from_date = datetime(from_date.year, from_date.month, from_date.day, 0, 1)
        self.from_date = from_date

        to_date = (
            from_date.replace(hour=23, minute=59)
            if to_date is None
            else datetime(to_date.year, to_date.month, to_date.day, 0, 1)
        )
        self.to_date = to_date
        self.credentials = credentials

    def from_date_as_str(self):  # Probably we dont need this
        return (
            datetimes.datetime_to_str(self.from_date)
            if isinstance(self.from_date, datetime)
            else self.from_date
        )

    def to_date_as_str(self):  # Probably we dont need this
        return (
            datetimes.datetime_to_str(self.to_date)
            if isinstance(self.to_date, datetime)
            else self.to_date
        )


def get_events_from_custom_layer(layer_id, meetings_request):
    # type: (int, MeetingsRequest) -> list
    from_date = meetings_request.from_date_as_str()
    to_date = meetings_request.to_date_as_str()

    if to_date < from_date:
        raise ValueError('to_date < from_date')

    params = {
        'actorUid': meetings_request.credentials.uid,
        'targetUid': meetings_request.subject_uid,
        'from': from_date,
        'to': to_date,
        'layerId': layer_id
    }
    response = request_to_calendar(
        method='get-events.json',
        params=params,
        credentials=meetings_request.credentials,
    )

    return response['events'] if response else None


def get_meetings(meetings_request):
    from_date = meetings_request.from_date_as_str()
    to_date = meetings_request.to_date_as_str()

    if to_date < from_date:
        raise ValueError('to_date < from_date')

    params = {
        'actorUid': meetings_request.credentials.uid,
        'targetUid': meetings_request.subject_uid,
        'from': from_date,
        'to': to_date,
        'layerToggledOn': 'true',
    }
    response = request_to_calendar('get-events.json', params, meetings_request.credentials)
    if response is None:
        return None
    for event in response['events']:
        event['url'] = '{}event/?event_id={}'.format(settings.CALENDAR_URL, event['id'])
        for resource in event.get('resources', []):
            try:
                room_info = staff.get_meeting_room_by_email(resource['email'])
            except Exception as e:
                logger.info('get_meeting_room_by_email raised exception %s', e)
                resource['not_found'] = True
            else:
                resource['not_found'] = False
                resource['id'] = room_info['id']
                resource['floor'] = room_info['floor']['id']
                resource['office_code'] = room_info['floor']['office']['code']
    return response['events']


def _find_available_resources_of_fixed_type(from_date, to_date, room_type):
    params = {
        'uid': settings.ROBOT_UID,
        'limit': 5
    }
    data = {
        'start': from_date,
        'end': to_date,
        'filter': room_type
    }
    response = post_request_to_calendar(
        'find-available-resources',
        params,
        data,
        CalendarRequestCredentials.uhura_credentials(),
    )
    available_resources = response['resources']
    if len(available_resources) == 1 and available_resources[0]['availability']:
        return []
    else:
        return [resource['name'] for resource in available_resources]


def find_available_resources(from_date, to_date, seats):  # Not sure if it's used
    room_types_dict = {
        1: 'small',
        2: 'medium',
        3: 'large'
    }
    room_type_number = 1
    if seats > settings.MEDIUM_ROOM_MAX_SEATS:
        room_type_number = 3
    elif seats > settings.SMALL_ROOM_MAX_SEATS:
        room_type_number = 2

    if isinstance(from_date, datetime):
        from_date = datetimes.datetime_to_str(from_date)
    if isinstance(to_date, datetime):
        to_date = datetimes.datetime_to_str(to_date)

    for i in range(room_type_number, 4):
        rooms = _find_available_resources_of_fixed_type(from_date, to_date, room_types_dict[room_type_number])
        if rooms:
            return rooms

    return 'Нет свободных переговорок'
