# coding: utf-8

from __future__ import unicode_literals

import collections
from datetime import date
import dateparser
import json
import operator
import os

from django.conf import settings
import logging

import pandas as pd

from uhura.external import intranet


logger = logging.getLogger(__name__)
FORMS_REQUESTS_TIMEOUT = 15


OPERATORS_MAPPING = {
    'and': operator.and_,
    'or': operator.or_
}


CONDITIONS_MAPPING = {
    'eq': operator.eq,
    'neq': operator.ne
}


class FilledError(Exception):
    pass


class NotFilledError(Exception):
    def __init__(self, field):
        self.field = field
        super(NotFilledError, self).__init__()


class SubmitError(Exception):
    pass


class SuccessfulSubmitError(Exception):
    pass


class SuggestNotFoundError(Exception):
    pass


class SuggestNotPerfectlyMatchedError(Exception):
    def __init__(self, suggested_items):
        self.suggested_items = suggested_items
        super(SuggestNotPerfectlyMatchedError, self).__init__()


class WrongInputError(Exception):
    pass


class BaseInput(object):
    @staticmethod
    def _preprocess_text(text):
        try:
            # use \n to create <table>
            return '\\n\\n'.join(['\\n'.join([' '.join(x) for x in df.as_matrix()]) for df in pd.read_html(text)])
        except ValueError:
            return text

    def __init__(self, field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice):
        self._answer = []
        self._id = field_id
        self._label = self._preprocess_text(label)
        self._help_text = self._preprocess_text(help_text)
        self._show_conditions = show_conditions
        self._is_allow_multiple_choice = is_allow_multiple_choice
        self._is_required = is_required
        self._is_skipped = False

    def try_autofill(self, person_data):
        if '__autofill__' not in self._id:
            return False

        value_name = self._id.rsplit('__autofill__', 1)[-1]
        if value_name == 'login':
            value = person_data['login']
        elif value_name == 'name':
            value = person_data['name']['first']['ru'] + ' ' + person_data['name']['last']['ru']
        else:
            raise NotImplementedError('Not implemented autofill value type %s' % value_name)

        self.set_answer(value)
        return True

    def has_answer(self):
        return len(self._answer) > 0

    @property
    def id(self):
        return self._id

    def is_required(self):
        return self._is_required

    def is_skipped(self):
        return self._is_skipped

    def is_allow_multiple_choice(self):
        return self._is_allow_multiple_choice

    def _preprocess_answer_item(self, x):
        return x

    def _preprocess_answer_item_text(self, x):
        return self._preprocess_answer_item(x)

    def get_single_answer(self):
        if not self._answer:
            return None

        assert len(self._answer) == 1
        return self.get_answer()

    def get_answer(self):
        return ','.join(map(self._preprocess_answer_item, self._answer))

    def get_answer_text(self):
        return ', '.join(map(self._preprocess_answer_item_text, self._answer))

    def get_label(self):
        return self._label

    def get_help_text(self):
        return self._help_text

    def get_options(self):
        pass

    def get_show_conditions(self):
        return self._show_conditions

    def check_show_conditions(self, form):
        if not self._show_conditions:
            return True

        for show_conditions in self._show_conditions:
            result = True
            for show_condition in show_conditions:
                operator = show_condition['operator']
                condition = show_condition['condition']
                if operator not in OPERATORS_MAPPING:
                    raise NotImplementedError('Not implemented operator %s' % operator)
                if condition not in CONDITIONS_MAPPING:
                    raise NotImplementedError('Not implemented condition %s' % condition)

                condition_field_value = show_condition['field_value']
                real_field_value = form[show_condition['field']].get_single_answer()
                cur_result = CONDITIONS_MAPPING[condition](real_field_value, condition_field_value)
                result = OPERATORS_MAPPING[operator](result, cur_result)

            if result:
                return True

        return False

    def set_answer(self, answer):
        if self._is_allow_multiple_choice:
            self._answer.append(answer)
        elif not self.has_answer():
            self._answer.append(answer)
            self._is_skipped = True
        else:
            raise ValueError('Answer was already set')

    def skip(self):
        if not self._is_required:
            self._is_skipped = True
        else:
            if self._is_allow_multiple_choice:
                if self.has_answer():
                    self._is_skipped = True
                else:
                    raise ValueError('Answer was not set')
            else:
                raise ValueError('This field doesn\'t allow multiple choice')

    def destroy(self):
        pass


class CheckBoxInput(BaseInput):
    def set_answer(self, answer):
        if answer.lower() == 'да':
            answer = 'True'
        elif answer.lower() == 'нет':
            answer = 'False'
        else:
            raise WrongInputError('CheckBoxInput must be bool')

        super(CheckBoxInput, self).set_answer(answer)

    def _preprocess_answer_item_text(self, x):
        if x == 'True':
            return 'Да'
        else:
            return 'Нет'

    def get_options(self):
        return ['Да', 'Нет']


class DateInput(BaseInput):
    def __init__(self, field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice, **kwargs):
        super(DateInput, self).__init__(
            field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice
        )

    def _preprocess_answer_item(self, x):
        return x.strftime('%Y-%m-%d')

    def set_answer(self, answer):
        if isinstance(answer, basestring):
            answer = dateparser.parse(answer, languages=['ru'])
        if not isinstance(answer, date):
            raise WrongInputError('DateInput answer must be a date')
        super(DateInput, self).set_answer(answer)


class EmptyInput(BaseInput):
    def __nonzero__(self):
        return False

    def has_answer(self):
        return False

    def is_skipped(self):
        return True


class FileInput(BaseInput):
    def __init__(self, field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice, **kwargs):
        super(FileInput, self).__init__(
            field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice
        )
        self._is_allow_multiple_choice = True
        assert not is_required

    def get_answer(self):
        return self._answer

    def get_answer_text(self):
        return 'Файлов загружено: %d' % len(self._answer)

    def destroy(self):
        for (file_name, _) in self._answer:
            try:
                os.remove(file_name)
            except OSError:  # file does not exist because it has not downloaded yet
                pass


class SelectInput(BaseInput):
    def __init__(self, field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice, **kwargs):
        self._options = kwargs['options']
        super(SelectInput, self).__init__(
            field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice
        )

    def _preprocess_answer_item(self, x):
        return self._options[x]['id']

    def _preprocess_answer_item_text(self, x):
        return self._options[x]['text']

    def get_options(self):
        return [x['text'] for x in self._options]

    def get_options_dict(self):
        return self._options

    def set_answer(self, answer):
        try:
            answer = [x['text'].lower() for x in self._options].index(answer.lower())
        except ValueError:
            raise WrongInputError('Value %s is not in options [%r]' % (answer.lower(), self._options))
        super(SelectInput, self).set_answer(answer)


class SuggestInput(BaseInput):
    def __init__(self, field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice, **kwargs):
        self.uri = kwargs['uri']
        self._answer_text = []
        super(SuggestInput, self).__init__(
            field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice
        )

    def _preprocess_suggest_query(self, query):
        return query.replace('(', '').replace(')', '')

    def _call_suggest(self, query):
        result = []
        url = '%s?suggest=%s' % (self.uri, self._preprocess_suggest_query(query))
        while url:
            response = intranet.get_request(url, timeout=FORMS_REQUESTS_TIMEOUT)
            if response is None:
                raise ValueError()
            url = response.get('next')
            result += response['results']
        return result

    def _preprocess_answer_item_text(self, x):
        return self._answer_text[self._answer.index(x)]

    def set_answer(self, answer):
        suggest_response = self._call_suggest(answer)
        if not suggest_response:
            raise SuggestNotFoundError()
        elif len(suggest_response) == 1:
            super(SuggestInput, self).set_answer(suggest_response[0]['id'])
            self._answer_text.append(suggest_response[0]['text'])
            return None
        else:
            raise SuggestNotPerfectlyMatchedError([x['text'] for x in suggest_response])

    def get_answer_text(self):
        return ', '.join(self._answer_text)


class TextInput(BaseInput):
    def __init__(self, field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice, **kwargs):
        super(TextInput, self).__init__(
            field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice
        )

    def set_answer(self, answer):
        if not isinstance(answer, basestring):
            raise WrongInputError('TextInput answer must be str or unicode')
        super(TextInput, self).set_answer(answer)


class NumberInput(BaseInput):
    def __init__(self, field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice, **kwargs):
        if kwargs['min'] is not None and kwargs['max'] is not None:
            min = int(kwargs['min'])
            max = int(kwargs['max'])
            self._allowed_range = range(min, max + 1)
            help_text += '\\n\\nЗначение должно быть от {} до {}'.format(min, max)
        super(NumberInput, self).__init__(
            field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice
        )

    def _preprocess_answer_item(self, x):
        return str(x)

    def validate_input(self, value):
        if self._allowed_range and value not in self._allowed_range:
            raise WrongInputError('Answer should be in [%r]' % self._allowed_range)

    def set_answer(self, answer):
        try:
            answer = int(answer)
        except ValueError:
            raise WrongInputError('Answer must be an integer')
        self.validate_input(answer)
        super(NumberInput, self).set_answer(answer)


class Form(object):
    INPUTS_MAPPING = {
        'CheckboxInput': CheckBoxInput,
        'list': SelectInput,
        'select': SelectInput,
        'DateInput': DateInput,
        'MultiFileInput': FileInput,
        'EmptyWidget': EmptyInput,
        'NumberInput': NumberInput,
        'Textarea': TextInput,
        'TextInput': TextInput
    }

    def __init__(self, form_id, json_str=None):
        self._form_id = form_id
        if not json_str:
            self._json = intranet.get_request(
                url=settings.FORM_BASE_URL.format(form_id),
                raw_response=True,
                timeout=FORMS_REQUESTS_TIMEOUT
            )
        else:
            self._json = json_str.encode('utf-8')

        fields = json.loads(self._json, object_pairs_hook=collections.OrderedDict)['fields']

        self._inputs = collections.OrderedDict()
        for field_id, field in fields.iteritems():
            is_allow_multiple_choice = field.get('is_allow_multiple_choice', False)
            is_required = field.get('is_required', False)
            widget_type = field['widget']
            label = field['label']
            help_text = field['help_text']
            show_conditions = field.get('show_conditions', None)
            if show_conditions is None:
                show_conditions = field.get('other_data', {}).get('show_conditions', None)

            try:
                input_class = Form.INPUTS_MAPPING[widget_type]
            except KeyError:
                logger.warning('Not implemented widget_type %s' % widget_type)
                continue

            additional_params = {}
            if input_class == SelectInput:
                if field['data_source'].get('uri'):
                    additional_params['uri'] = field['data_source']['uri']
                    input_class = SuggestInput
                else:
                    additional_params['options'] = field['data_source']['items']
            elif input_class == NumberInput:
                attrs = field.get('tags', [{}])[0].get('attrs', {})
                additional_params['min'] = attrs.get('min', None)
                additional_params['max'] = attrs.get('max', None)
            input = input_class(
                field_id, label, help_text, show_conditions, is_required, is_allow_multiple_choice, **additional_params
            )
            self._inputs[field_id] = input

    def __getitem__(self, item):
        return self._inputs[item]

    def __iter__(self):
        return self._inputs.iteritems()

    def __str__(self):
        result = []
        for field in self._inputs.itervalues():
            if field.has_answer():
                result.append('{}: {}'.format(field.get_label(), field.get_answer_text()))
        return '\\n'.join(result).encode('utf-8')

    def fill(self, answers, person_data, raise_filled=True):
        for (field_id, field) in self._inputs.iteritems():
            if answers[field_id]:
                for answer in answers[field_id]:
                    if answer == '_skip':
                        field.skip()
                    else:
                        field.set_answer(answer)
                if not field.is_skipped():
                    raise NotFilledError(field)
            elif field.try_autofill(person_data):
                answers[field_id].append(field.get_answer())
            elif field.check_show_conditions(self):
                raise NotFilledError(field)

        if raise_filled:
            raise FilledError()

    def get_json(self):
        return self._json.decode('utf-8')

    def get_final_text(self):
        data = intranet.get_request(settings.SURVEY_BASE_URL.format(self._form_id), timeout=FORMS_REQUESTS_TIMEOUT)
        if data is None:
            return None
        return '{}\n{}'.format(data['texts']['successful_submission_title'],  data['texts']['successful_submission'])

    def submit(self, login, connector):
        multipart_form_data = []
        for (field_id, field) in self._inputs.iteritems():
            if field.has_answer():
                if isinstance(field, FileInput):
                    new_file_names = []
                    for (file_name, file_id) in field.get_answer():
                        file_name = connector.download_file(file_id, file_name)
                        new_file_names.append(file_name)
                        response = intranet.post_request(
                            settings.FORM_FILES_URL,
                            files={'file': (file_name, open(file_name, 'rb'))},
                            headers={'X-FRONTEND-AUTHORIZATION': settings.FORMS_AUTH_TOKEN},
                            timeout=FORMS_REQUESTS_TIMEOUT
                        )
                        if response is None:
                            return None
                        multipart_form_data.append((field_id, ('', response['id'])))
                    field._answer = [(new_file_names[i], ans[1]) for (i, ans) in enumerate(field.get_answer())]
                else:
                    multipart_form_data.append((field_id, ('', field.get_answer())))

        data = {}
        data['source_request'] = json.dumps({
            'headers': {
                'X-UHURA-REQUESTER': login
            },
        })

        self.release_all_resources()
        response = intranet.post_request(
            url=settings.FORM_BASE_URL.format(self._form_id),
            files=multipart_form_data,
            data=data,
            timeout=FORMS_REQUESTS_TIMEOUT
        )
        if response is None:
            raise SubmitError()
        else:
            raise SuccessfulSubmitError

    def release_all_resources(self):
        for field in self._inputs.itervalues():
            field.destroy()
