# coding: utf-8
from __future__ import unicode_literals

from datetime import timedelta
import urlparse

from django.conf import settings

from uhura.external import intranet
from uhura.external import staff
from uhura.utils import datetimes


FORMAT_DICT = {
    True: 'Да',
    False: 'Нет',
    'illness': 'Болезнь',
    'vacation': 'Отпуск',
    'maternity': 'Уход за ребенком',
    'learning': 'Обучение',
    'trip': 'Командировка',
    'conference_trip': 'Конференция',
    'absence': 'Отсутствие',
    'paid_day_off': 'Отгул'
}

REVERSED_FORMAT_DICT = {v.lower(): k for (k, v) in FORMAT_DICT.iteritems()}


POSSIBLE_WORKFLOWS = ['absence', 'trip', 'conference_trip', 'learning', 'vacation', 'paid_day_off', 'illness']
IMPLEMENTED_GAP_TYPES = ['отсутствие', 'обучение', 'отгул', 'болезнь', 'отпуск']
FULL_DAY_GAP_TYPES = ['болезнь', 'отгул', 'отпуск']


class GapError(Exception):
    pass


class GapIntersectionError(GapError):
    pass


class GapWrongDatesError(GapError):
    pass


class GapPaidDayOffCountError(GapError):
    pass


def cast_timestamps_to_readable_format(gaps):
    if gaps:
        for i in range(len(gaps)):
            gaps[i]['date_from'] = datetimes.datetime_str_to_readable_format(gaps[i]['date_from'])
            gaps[i]['date_to'] = datetimes.datetime_str_to_readable_format(gaps[i]['date_to'])


def get_request_to_gap(method, params):
    url = urlparse.urljoin(settings.GAP_API_URL, method)
    return intranet.get_request(url, params=params)


def post_request_to_gap(method, data):
    url = urlparse.urljoin(settings.GAP_API_URL, method)
    return intranet.post_request(url, data=data)


def _create_gap_utc(login, from_date, to_date, work_in_absence, message, workflow, full_day, has_sicklist=None):
    to_datetime = datetimes.datetime_to_str(to_date)
    from_datetime = datetimes.datetime_to_str(from_date)

    if message is None:
        message = 'Отсутствие создано с помощью telegram бота'

    if workflow not in POSSIBLE_WORKFLOWS:
        raise NotImplementedError('not implemented workflow %s' % workflow)

    if workflow == 'paid_day_off':
        duration = (to_date - from_date).days + 1
        max_duration = staff.get_paid_day_off_count(login)
        if max_duration is None:
            raise GapError('staff unavailable')
        if max_duration < duration:
            raise GapPaidDayOffCountError('paid_day_off limit exceeded')

    data = {
        'workflow': workflow,
        'person_login': login,
        'date_from': from_datetime,
        'date_to': to_datetime,
        'work_in_absence': work_in_absence,
        'comment': message,
        'full_day': full_day
    }
    if has_sicklist:
        data['has_sicklist'] = has_sicklist

    return post_request_to_gap(method='gap_create', data=data)


def create_gap(login, from_date, to_date=None, work_in_absence=False, message=None, workflow='absence',
               has_sicklist=False):
    user_tz = staff.get_timezone_by_login(login)
    if isinstance(from_date, basestring):
        from_date = datetimes.str_to_datetime(from_date)
    if isinstance(to_date, basestring):
        to_date = datetimes.str_to_datetime(to_date)

    if to_date is None:
        from_date = from_date.replace(hour=0, minute=1)
        to_date = from_date.replace(hour=23, minute=59)
        full_day = True
    else:
        if not any((to_date.hour, to_date.minute, to_date.second)):
            to_date = to_date + timedelta(hours=23, minutes=59)
            full_day = True
        else:
            full_day = False

        if not any((from_date.hour, from_date.minute, from_date.second)):
            from_date = from_date + timedelta(minutes=1)
        else:
            full_day = False

    gaps = get_gaps(login, from_date, to_date)
    if gaps:
        raise GapIntersectionError('gaps intersection')

    if not full_day:
        from_date = datetimes.convert_datetime(from_date, user_tz, 'UTC')
        to_date = datetimes.convert_datetime(to_date, user_tz, 'UTC')
    if message is None:
        message = 'Отсутствие создано с помощью telegram бота'
    if workflow.lower() in REVERSED_FORMAT_DICT:
        workflow = REVERSED_FORMAT_DICT[workflow.lower()]
    if workflow not in POSSIBLE_WORKFLOWS:
        raise NotImplementedError('not implemented workflow %s' % workflow)
    return _create_gap_utc(login, from_date, to_date, work_in_absence, message, workflow, full_day, has_sicklist)


def _get_gaps_utc(login, from_date, to_date=None):
    if to_date:
        if to_date < from_date:
            raise GapWrongDatesError('to_date < from_date')
        from_date = datetimes.datetime_to_str(from_date + timedelta(seconds=1))
        to_date = datetimes.datetime_to_str(to_date + timedelta(seconds=1))
    else:
        from_date = datetimes.datetime_to_str(from_date + timedelta(seconds=1))
        to_date = datetimes.str_to_datetime(from_date).replace(hour=23, minute=59)

    params = {
        'person_login': login,
        'date_from': from_date,
        'date_to': to_date
    }
    response = get_request_to_gap(method='gaps_find', params=params)
    if response is None:
        return None
    return response['gaps']


def get_gaps(login, from_date, to_date):
    user_tz = staff.get_timezone_by_login(login)
    if isinstance(from_date, basestring):
        from_date = datetimes.str_to_datetime(from_date)
    if isinstance(to_date, basestring):
        to_date = datetimes.str_to_datetime(to_date)

    from_date_utc = datetimes.convert_datetime(from_date, user_tz, 'UTC')
    to_date_utc = datetimes.convert_datetime(to_date, user_tz, 'UTC')

    gaps = _get_gaps_utc(login, from_date_utc, to_date_utc)

    if gaps:
        result = []
        for gap in gaps:
            if gap.get('full_day'):
                gap_start = datetimes.str_to_datetime(gap['date_from']) + timedelta(minutes=1)
                gap_end = datetimes.str_to_datetime(gap['date_to']) - timedelta(minutes=1)
                if (min(to_date, gap_end) - max(from_date, gap_start)).total_seconds() > 0:
                    gap['date_from'] = datetimes.datetime_to_str(gap_start)
                    gap['date_to'] = datetimes.datetime_to_str(gap_end)
                    result.append(gap)
            else:
                gap['date_from'] = datetimes.convert_utc_to_usertz(gap['date_from'], user_tz)
                gap['date_to'] = datetimes.convert_utc_to_usertz(gap['date_to'], user_tz)
                result.append(gap)
    else:
        result = gaps
    return result
