# coding: utf-8

from __future__ import unicode_literals

import logging

from django.conf import settings
import requests
import waffle

from ids.utils import https as ids_https


logger = logging.getLogger(__name__)
INTRANET_TIMEOUT = 5
MAX_REQUEST_RETRIES = 3


def get_request(url, params=None, auth_headers=None, headers=None, retries=MAX_REQUEST_RETRIES, raw_response=False,
                timeout=INTRANET_TIMEOUT, verify=True, raise_exception=False):
    last_exc = None
    if verify:
        ca_bundle_path = ids_https.get_ca_bundle_path()
        verify = ca_bundle_path or False
    if headers is None:
        headers = {}

    headers.update(auth_headers or settings.AUTH_HEADERS)

    response = None
    if waffle.switch_is_active('not_use_intranet_retries') and retries > 0:
        retries = 1
    try:
        response = requests.get(
            url=url,
            params=params,
            headers=headers,
            timeout=timeout,
            verify=verify,
        )
        response.raise_for_status()
        if raw_response:
            return response.content
        else:
            return response.json()
    except requests.exceptions.ConnectTimeout as last_exc:
        logger.error('intranet.get_request connect timeout: %s', last_exc)
    except requests.exceptions.ReadTimeout as last_exc:
        if retries > 0:
            return get_request(
                url,
                params=params,
                headers=headers,
                retries=retries - 1,
                raw_response=raw_response,
                timeout=timeout,
                verify=verify
            )
        else:
            logger.error(
                'GET-retries exceeded for url %s, params %r, exception %s',
                url, params, last_exc
            )
    except requests.exceptions.HTTPError as last_exc:
        if response and response.status_code == 404:
            logger.error('Url not found: %s', response.url)
        else:
            if retries > 0:
                return get_request(
                    url,
                    params=params,
                    headers=headers,
                    retries=retries - 1,
                    raw_response=raw_response,
                    timeout=timeout,
                    verify=verify
                )
            elif response:
                logger.error(
                    'Could not access URL %s, received HTTP %d %s',
                    response.url,
                    response.status_code,
                    response.reason,
                    exc_info=True
                )
            else:
                logger.exception(
                    'GET-retries exceeded for url %s, params %r, exception %s',
                    url, params, last_exc
                )
    except ValueError:
        logger.error('Bad json response URL %s', response.url)

    if last_exc and raise_exception:
        raise last_exc

    return None


def post_request(url, params=None, data=None, auth_headers=None, headers=None, files=None, retries=MAX_REQUEST_RETRIES,
                 timeout=INTRANET_TIMEOUT, raw=False, verify=True, raise_exception=False, raw_response=False):
    if verify:
        ca_bundle_path = ids_https.get_ca_bundle_path()
        verify = ca_bundle_path or False
    if headers is None:
        headers = {}

    headers.update(auth_headers or settings.AUTH_HEADERS)

    response = None
    if waffle.switch_is_active('not_use_intranet_retries') and retries > 0:
        retries = 1
    try:
        if files or raw:
            response = requests.post(
                url=url,
                params=params,
                data=data,
                files=files,
                headers=headers,
                timeout=timeout,
                verify=verify,
            )
        else:
            response = requests.post(
                url=url,
                params=params,
                json=data,
                headers=headers,
                timeout=timeout,
                verify=verify,
            )
        response.raise_for_status()
        if raw_response:
            return response.content
        else:
            return response.json()
    except requests.exceptions.ConnectTimeout as e:
        logger.error('intranet.post_request connect timeout: %s', e)
    except requests.exceptions.ReadTimeout as e:
        logger.error('intranet.post_request read timeout: %s', e)
    except requests.exceptions.HTTPError as e:
        if response and response.status_code == 404:
            logger.error('Url not found: %s', response.url)
        elif response and response.status_code == 500:
            logger.error('Internal Server Error: %s, url: %s', response.text, response.url)
        elif response:
            # Если мы получили ответ на POST запрос от сервера,
            # это значит что запрос обработан и ретраить нельзя
            logger.error(
                'Could not access URL %s, received HTTP %d %s',
                response.url,
                response.status_code,
                response.reason,
                exc_info=True
            )
        else:
            if retries > 0:
                return post_request(
                    url,
                    params=params,
                    data=data,
                    headers=headers,
                    files=files,
                    retries=retries - 1,
                    timeout=timeout,
                    verify=verify,
                    raise_exception=raise_exception,
                    raw_response=raw_response,
                )
            else:
                logger.exception(
                    'POST-retries exceeded for url %s, params %r, data %r, files %r',
                    url, params, data, files
                )
    except ValueError as e:
        logger.error('Bad json response URL %s, content %s', response.url, response.content)
    if raise_exception:
        raise e
    else:
        return None
