# coding: utf-8

from __future__ import unicode_literals

import json
from urlparse import urljoin

from django.conf import settings

from ids.registry import registry
import ids

from uhura.external import intranet
from uhura.lib import cache_storage


persons = registry.get_repository(
    'staff', 'person', user_agent='uhura', oauth_token=settings.ROBOT_TOKEN, retries=3
)
rooms = registry.get_repository(
    'staff', 'room', user_agent='uhura', oauth_token=settings.ROBOT_TOKEN, retries=3
)


@cache_storage.memoize_decorator(fresh_time=settings.STAFF_CACHE_FRESH_TIME)
def get_uid_by_login(login):
    return persons.get_one(lookup={'login': login, '_fields': 'uid'})['uid']


@cache_storage.memoize_decorator(fresh_time=settings.STAFF_CACHE_FRESH_TIME, cache_none=False)
def get_timezone_by_uid(uid):
    return persons.get_one(lookup={'uid': uid, '_fields': 'environment.timezone'})['environment']['timezone']


@cache_storage.memoize_decorator(fresh_time=settings.STAFF_CACHE_FRESH_TIME, cache_none=False)
def get_timezone_by_login(login):
    return persons.get_one(lookup={'login': login, '_fields': 'environment.timezone'})['environment']['timezone']


@cache_storage.memoize_decorator(fresh_time=settings.STAFF_CACHE_FRESH_TIME)
def get_office_by_login(login):
    return persons.get_one(lookup={'login': login, '_fields': 'location.office'})['location']['office']


@cache_storage.memoize_decorator(fresh_time=settings.STAFF_CACHE_FRESH_TIME)
def get_meeting_room_by_email(email):
    return rooms.get_one(lookup={'name.exchange': email.split('@')[0]})


def find_main_phone(phones_list):
    main_phone = None
    if phones_list:
        main_phone = phones_list[0]['number']
        for phone in phones_list:
            if phone.get('is_main', False):
                main_phone = phone['number']
                break
    return main_phone


def find_telegram_accounts(accounts):
    return {account['value'] for account in accounts if account['type'] == 'telegram'}


def get_person_info(login):
    response = persons.get_one(lookup={
        'login': login,
        '_fields': 'location.table,phones,name,accounts,work_phone,personal.gender'
    })
    try:
        first_name = response['name']['first']['ru']
    except KeyError:
        first_name = ''
    try:
        last_name = response['name']['last']['ru']
    except KeyError:
        last_name = ''

    gender = response['personal']['gender'][0]
    name = first_name + ' ' + last_name
    table = response['location']['table']
    main_phone = find_main_phone(response['phones'])
    telegram_accounts = find_telegram_accounts(response['accounts'])
    work_phone = response['work_phone']
    person_info = {
        'name': name,
        'login': login,
        'phone': main_phone,
        'telegram_username': telegram_accounts,
        'table': table,
        'work_phone': work_phone,
        'gender': gender,
        '_telegram_username_folded': ', '.join(['@' + login for login in telegram_accounts])
    }
    return person_info


@cache_storage.memoize_decorator(fresh_time=settings.STAFF_CACHE_FRESH_TIME, cache_none=False)
def get_person_data_by_uid(uid):
    try:
        return persons.get_one(lookup={'uid': uid})
    except ids.exceptions.BackendError as e:
        if e.status_code == 404:
            return None
        else:
            raise


@cache_storage.memoize_decorator(fresh_time=settings.STAFF_CACHE_FRESH_TIME, cache_none=False)
def get_person_data_by_telegram_username(username):
    if not username:
        return None
    try:
        return persons.get_one(lookup={
            '_query': (
                'accounts == match({"type": "telegram", "value_lower": "%s"}) '
                'and official.is_dismissed == False'
            ) % username.lower(),
            '_fields': 'groups.group.id,'
                       'groups.group.ancestors.id,'
                       'login,'
                       'name,'
                       'official.affiliation,'
                       'location.office,'
                       'uid'
        })
    except ids.exceptions.BackendError as e:
        if e.status_code == 404:
            return None
        else:
            raise


@cache_storage.memoize_decorator(fresh_time=settings.STAFF_CACHE_FRESH_TIME, cache_none=False)
def get_person_data_by_userphone(phone):
    if not phone:
        return None
    phone = phone.replace('+', '')
    phone = '^\+%s$' % '[ -]?'.join(phone)
    try:
        return persons.get_one(lookup={
            '_query': 'phones.number == regex("%s") and official.is_dismissed == False' % phone
        })
    except ids.exceptions.BackendError as e:
        if e.status_code == 404:
            return None
        else:
            raise


def get_food_balance(login):
    return intranet.get_request(settings.FOOD_BALANCE_API_URL.format(login=login))['target']


def get_profile(login):
    return intranet.get_request(urljoin(settings.PROFILE_API_URL, login))['target']


def get_last_online(login):
    response = intranet.get_request(settings.LAST_ONLINE_URL + login, raw_response=True)
    data_json = json.loads(''.join(response.split("(")[1:])[:-2])
    try:
        return data_json['content'][1]['content'][:-1]
    except KeyError:
        return 'Нет данных о последней активности сотрудника'


def get_paid_day_off_count(login):
    response = intranet.get_request(urljoin(settings.PROFILE_API_URL, login))
    try:
        return response['target'].get('paid_day_off', 0)
    except (KeyError, TypeError):
        return None
