# coding: utf-8

from __future__ import unicode_literals

from itertools import islice
import logging
import re

from startrek_client import Startrek
from startrek_client.exceptions import StartrekError, NotFound, Forbidden

from django.conf import settings

from uhura.external.intranet import MAX_REQUEST_RETRIES, INTRANET_TIMEOUT


logger = logging.getLogger(__name__)


client = Startrek(
    useragent='uhura',
    base_url=settings.ST_API_URL,
    token=settings.ROBOT_TOKEN,
    retries=MAX_REQUEST_RETRIES,
    timeout=INTRANET_TIMEOUT
)
possible_statuses = ['inProgress', 'inReview', 'open', 'closed']
ticket_regexp = re.compile('[A-Z]+-\d+')


class WrongTicketKey(Exception):
    pass


def check_queue(queue_name):
    try:
        client.queues[queue_name]
    except NotFound:
        return None
    except Forbidden:
        return False
    else:
        return True


def get_ticket(key):
    if not ticket_regexp.match(key):
        raise WrongTicketKey('Wrong ticket key format %s' % key)
    try:
        ticket = client.issues[key]
        return {'description': ticket.description, 'summary': ticket.summary, 'author': ticket.createdBy.login}
    except (Forbidden, NotFound):
        return None


def get_tickets(login, status=None, tickets_number=25):
    filter = {'assignee': login}
    if status:
        assert status in possible_statuses
        filter['status'] = status
    try:
        issues = client.issues.find(
            filter=filter,
            per_page=tickets_number
        )
    except StartrekError:
        logger.exception('Can\'t get tickets')
        return None
    else:
        return [
            {
                'link': issue.self.replace('-api', '').replace('/v2/issues', ''),
                'summary': issue.summary
            }
            for issue in islice(issues, tickets_number)
        ]


def create_ticket(login, queue, summary, description=None, attachments=None, tags=None):
    try:
        if tags:
            tags.append('created_with_Uhura')
        else:
            tags = ['created_with_Uhura']
        return client.issues.create(
            queue=queue,
            summary=summary,
            description=description,
            createdBy=login,
            tags=tags,
            attachments=attachments
        ).self.replace('-api', '').replace('/v2/issues', '')
    except Exception:
        logger.exception('Can\'t create ticket')
        return None
