# coding: utf-8
from __future__ import unicode_literals

import re

from django.conf import settings

from uhura.external import intranet
from uhura.external import staff

import logging


logger = logging.getLogger(__name__)
letters_map = {
    'а': 'a',
    'в': 'b',
    'е': 'e',
    'к': 'k',
    'м': 'm',
    'н': 'h',
    'о': 'o',
    'р': 'p',
    'с': 'c',
    'т': 't',
    'у': 'y',
    'х': 'x'
}


def _cut_name(name):
    if len(name) > 6:
        return name[:-3]
    elif len(name) > 4:
        return name[:-2]
    else:
        return name[:-1]


LAYER_PEOPLE = 'people'
LAYER_MEETING_ROOM = 'invite'
LAYER_CLINICS = 'clinics'
LAYERS_PARAMS = {
    LAYER_PEOPLE: {
        'people.per_page': 11,
        'people.query': 'i_is_robot:0 & i_is_dismissed:0'
    }
}


def _call_suggest(text, layers, **kwargs):
    params = {
        'version': 2,
        'text': text,
        'layers': ','.join(layers)
    }
    for layer in layers:
        params.update(LAYERS_PARAMS.get(layer, {}))
    if 'people.query' in kwargs:
        params['people.query'] = '{} & ({})'.format(params['people.query'], kwargs.pop('people.query'))
    params.update(**kwargs)
    return intranet.get_request(url=settings.SUGGEST_URL, params=params)


def _find_person(name):
    return _call_suggest(name, [LAYER_PEOPLE])


def _find_exact_match(name, persons_list):
    for word in name.split():
        new_persons_list = []
        for person in persons_list:
            login = person['login']
            full_name = person['name'].replace('(', '').replace(')', '')
            if login.lower() == word.lower() or any([x.lower() == word.lower() for x in full_name.split()]):
                new_persons_list.append(person)
        persons_list = new_persons_list[:]
    return persons_list


def _format_plate(plate):
    res_plate = []
    for char in plate:
        if char.isdigit() or char.isalpha():
            if char.lower() in letters_map:
                res_plate.append(letters_map[char.lower()].upper())
            else:
                res_plate.append(char.upper())
    return ''.join(res_plate)


def _parse_persons_list(name, persons_list):
    if not persons_list:
        try:
            staff_login = staff.get_person_data_by_telegram_username(name)['login']
            return find_person(staff_login)
        except TypeError:
            splitted_name = name.split()
            cutted_name = ' '.join(map(_cut_name, splitted_name))
            result = _find_person(cutted_name)
            if result is None:
                return None, 'error'
            persons_list = result[LAYER_PEOPLE]['result']
        except Exception:
            return None, 'error'

    persons_list = [{'name': p['title'], 'login': p['login'], 'uid': p['uid']} for p in persons_list]
    exact_match = _find_exact_match(name, persons_list)
    if exact_match:
        persons_list = exact_match

    if len(persons_list) == 1:
        phrase_id = 'person_answer'
    elif len(persons_list) >= 10:
        phrase_id = 'person_specify'
    elif not persons_list:
        phrase_id = 'person_not_found'
    else:
        phrase_id = 'person_choose'
    return persons_list, phrase_id


def _parse_meeting_room_list(rooms_list):
    if not rooms_list:
        phrase_id = 'not_found'
    elif len(rooms_list) < 5:
        phrase_id = 'meeting_room_answer'
    else:
        phrase_id = 'meeting_room_specify'
    return rooms_list, phrase_id


def _clean_name(name):
    name = name.replace('@', '')
    if len(name.split()) > 2:
        raise ValueError
    return name


def find_meeting_room(name):
    results = _call_suggest(name, [LAYER_MEETING_ROOM])
    if results is None:
        return None, 'error'

    rooms_list = results[LAYER_MEETING_ROOM]['result']
    return _parse_meeting_room_list(rooms_list)


def find_person(name):
    name = _clean_name(name)

    results = _find_person(name)
    if results is None:
        return None, 'error'

    persons_list = results[LAYER_PEOPLE]['result']
    return _parse_persons_list(name, persons_list)


def find_meeting_room_or_table(name):
    result = _call_suggest(name, [LAYER_PEOPLE, LAYER_MEETING_ROOM])
    if result is None:
        return None, 'error'

    persons_list, persons_phrase_id = _parse_persons_list(name, result[LAYER_PEOPLE]['result'])
    rooms_list, rooms_phrase_id = _parse_meeting_room_list(result[LAYER_MEETING_ROOM]['result'])

    if rooms_list and persons_list:
        return persons_list + rooms_list, 'choose'
    elif persons_list:
        return persons_list, persons_phrase_id
    else:
        return rooms_list, rooms_phrase_id


def _parse_clinics_list(result):
    if not result['result']:
        phrase_id = 'not_found'
    else:
        phrase_id = 'answer'
    clinics_list = result['result']
    return clinics_list, result['pagination']['count'], phrase_id


def find_clinics(name, city):
    params = {}
    city_param_name = '%s.query' % LAYER_CLINICS
    if city:
        params.update({city_param_name: 'z_hr_address:(%s)' % city})

    results = _call_suggest(name, [LAYER_CLINICS], **params)
    if results is None:
        return None, 0, 'error'

    return _parse_clinics_list(results[LAYER_CLINICS])


def _create_plate_regexp(plate):
    exp = []
    for char in plate:
        if char.isdigit():
            exp.append(char)
        elif char.isalpha():
            char = char.lower()
            mapped_char = None
            if char in letters_map.keys():
                mapped_char = letters_map[char]
            else:
                for k, v in letters_map.iteritems():
                    if v == char:
                        mapped_char = k
                        break
            exp.append('[%s%s%s%s]' % (char, char.upper(), mapped_char, mapped_char.upper()))
    exp = '.*' + '.*'.join(exp) + '.*'
    return exp


def _parse_car_owners_list(pattern, results):
    if not results:
        return []

    owners_list = []
    plate_regexp = _create_plate_regexp(pattern)
    for result in results:
        bicycles_and_cars = result['cars'] + result['bicycles']
        for obj in bicycles_and_cars:
            obj_plate = obj['number']
            if re.match(plate_regexp, obj_plate, re.U):
                person = staff.get_person_data_by_uid(result['uid'])
                if person is None:
                    return None
                phones = person.get('phones')
                main_phone = staff.find_main_phone(phones)
                telegram = []
                for account in person['accounts']:
                    if account['type'] == 'telegram':
                        telegram.append(account['value'])
                owners_list.append({
                    'name': person['name']['first']['ru'] + ' ' + person['name']['last']['ru'],
                    'login': person['login'],
                    'plate': _format_plate(obj_plate),
                    'model': obj.get('model') or obj.get('description'),
                    'phone': main_phone,
                    'telegram_username': telegram
                })
    return owners_list


def find_car_owner(plate):
    patterns = [plate, re.sub('\s+', '', plate, flags=re.U)]
    params = {
        'allow_empty': '1'
    }
    for pattern in patterns:
        params['people.query'] = 'z_people_bicycle:({0}) | z_people_car:({0})'.format(pattern)
        response = _call_suggest('', [LAYER_PEOPLE], **params)
        if response is None:
            return None
        try:
            owners_list = _parse_car_owners_list(pattern, response['people']['result'])
        except Exception:
            logger.exception('find_car_owner raised exception')
            return None
        else:
            if owners_list:
                return owners_list
    return []
