# coding: utf-8
from __future__ import unicode_literals

import logging
from datetime import datetime, timedelta

import ids

from django.conf import settings

from uhura.external import calendar
from uhura.external import gap
from uhura.external import staff
from uhura.external import startrek
from uhura.lib import cache_storage
from uhura.utils import datetimes


logger = logging.getLogger(__name__)


def get_next_meeting(uid, credentials):
    # type: (str, calendar.CalendarRequestCredentials) -> list

    try:
        user_tz = staff.get_timezone_by_uid(uid)
    except Exception as e:
        user_tz = 'Europe/Moscow'
        logger.warning('get_timezone raised exception' + str(e))
    from_date = datetimes.convert_datetime(datetime.now(), 'Europe/Moscow', user_tz)
    to_date = from_date + timedelta(days=7)
    request = calendar.MeetingsRequest(uid, from_date, to_date, credentials)
    events = calendar.get_meetings(request)
    if events is None:
        return None
    else:
        next_event_start = None
        result = []
        for event in events:
            start = datetimes.str_to_datetime(event['startTs'])
            start = datetimes.convert_datetime(start, user_tz, user_tz)
            if next_event_start is None and start > from_date:
                next_event_start = start
                result.append(event)
            elif next_event_start and start == next_event_start:
                result.append(event)
    return result


def get_current_meeting(uid, credentials):
    # type: (str, calendar.CalendarRequestCredentials) -> list
    try:
        user_tz = staff.get_timezone_by_uid(uid)
    except Exception as e:
        user_tz = 'Europe/Moscow'
        logger.warning('get_timezone raised exception' + str(e))
    from_date = datetimes.convert_datetime(datetime.now(), 'Europe/Moscow', user_tz)
    to_date = from_date + timedelta(minutes=1)
    request = calendar.MeetingsRequest(uid, from_date, to_date, credentials)
    events = calendar.get_meetings(request)
    return events


def get_shuttle_schedule(layer_id):
    cached_value = cache_storage.get('shuttle_schedule_{}'.format(layer_id))
    if cached_value == cache_storage.EXPIRED_VALUE:
        dt = datetime.now()
        try:
            request = calendar.MeetingsRequest(
                settings.ROBOT_UID,
                dt,
                dt.replace(hour=23, minute=59),
                calendar.CalendarRequestCredentials.uhura_credentials(),
            )
            events = calendar.get_events_from_custom_layer(layer_id, request)[:3]

            if not events:
                if dt.weekday() in [4, 5, 6]:
                    dt = datetimes.next_weekday(dt, 0).replace(hour=0, minute=1)
                else:
                    dt = (dt + timedelta(days=1)).replace(hour=0, minute=1)

                request = calendar.MeetingsRequest(
                    settings.ROBOT_UID,
                    dt,
                    dt.replace(hour=23, minute=59),
                    calendar.CalendarRequestCredentials.uhura_credentials(),
                )
                events = calendar.get_events_from_custom_layer(layer_id, request)[:3]
            next_shuttle_time = datetimes.str_to_datetime(events[0]['startTs'])
        except (TypeError, ids.exceptions.BackendError):
            return None, None
        else:
            cache_storage.set(
                'shuttle_schedule_{}'.format(layer_id),
                (events, dt),
                (next_shuttle_time - datetime.now()).total_seconds()
            )
    else:
        events, dt = cached_value

    if dt.weekday() == datetime.now().weekday():
        dt = None
    elif dt.weekday() - datetime.now().weekday() == 1:
        dt = 'завтра'
    else:
        dt = 'в понедельник'

    schedule = [x['startTs'][11:16] for x in events]
    return schedule, dt


def get_meetings_by_date(meetings_request):
    # type: (calendar.MeetingsRequest) -> list
    events = calendar.get_meetings(meetings_request)

    if meetings_request.subject_uid != meetings_request.credentials.uid:
        login = meetings_request.credentials.person_login
        for i in range(len(events)):
            if not events[i].get('othersCanView') or events[i].get('primaryLayerClosed'):
                attendees_logins = [x.get('login') for x in events[i].get('attendees', [])]
                organizer_login = events[i].get('organizer', {}).get('login')
                events[i]['hidden_event'] = (login not in attendees_logins and login != organizer_login)

    return events


def get_person_info(login, credentials):
    try:
        person_info = staff.get_person_info(login)
    except ids.exceptions.BackendError as e:
        if e.status_code == 404:
            return None
        else:
            raise
    from_date = datetime.now()
    to_date = from_date + timedelta(hours=1)
    gaps = gap.get_gaps(login, from_date, to_date)
    if gaps:
        person_info['gap'] = gaps[0]
    else:
        uid = staff.get_uid_by_login(login)
        meetings_request = calendar.MeetingsRequest(uid, from_date, to_date, credentials)
        events = calendar.get_meetings(meetings_request)
        if events:
            person_info['event'] = events[0]
    return person_info


def get_top5_tickets(login):
    return startrek.get_tickets(login, tickets_number=5)


def get_tickets_inProgress(login):
    return startrek.get_tickets(login, status='inProgress', tickets_number=25)


def get_gaps(login, from_date, to_date=None):
    from_date = datetime(from_date.year, from_date.month, from_date.day, 0, 0, 1)
    if to_date is None:
        to_date = from_date.replace(hour=23, minute=59)
    else:
        to_date = datetime(to_date.year, to_date.month, to_date.day, 23, 59)
    return gap.get_gaps(login, from_date, to_date)


def work_from_home(login, date):
    return gap.create_gap(login, work_in_absence=True, from_date=date, message='Работаю из дома', workflow='absence')
