# coding: utf-8

from datetime import datetime

from django.conf import settings
from django.db import transaction
import waffle

from vins_core.dm import session as vins_session

from uhura import models
from uhura.lib.vins.transition import intents


MIN_TIMEOUT = 60


def get_valid_session(session_obj, timeout=None):
    if timeout is None or session_obj.session is None:
        return session_obj.session

    if intents.is_quest_intent(vins_session.Session.from_dict(session_obj.session).intent_name):
        return session_obj.session

    timeout = max(timeout, MIN_TIMEOUT)
    if (datetime.utcnow() - session_obj.updated_at.replace(tzinfo=None)).total_seconds() < timeout:
        return session_obj.session
    else:
        return None


class PostgressSessionStorage(vins_session.BaseSessionStorage):
    def _load(self, app_id, uuid, dialog_id):
        try:
            if waffle.switch_is_active('not_use_session_lock'):
                session_obj = models.Session.objects.get(app_id=app_id, uuid=uuid)
            else:
                session_obj = models.Session.objects.select_for_update(nowait=True).get(app_id=app_id, uuid=uuid)
        except transaction.TransactionManagementError:
            session_obj = models.Session.objects.get(app_id=app_id, uuid=uuid)
        except models.Session.DoesNotExist:
            return None

        return get_valid_session(session_obj, settings.SESSION_TIMEOUT)

    @transaction.atomic
    def _save(self, app_id, uuid, dialog_id, session):
        (session_obj, _) = models.Session.objects.get_or_create(app_id=app_id, uuid=uuid)
        session_obj.session = session
        session_obj.save()
