# coding: utf-8

from __future__ import unicode_literals

import logging
import os

from datetime import datetime

import yenv

from vins_sdk import connectors

from uhura.utils import hasher
from uhura.utils.datetimes import datetime_to_str


logger = logging.getLogger(__name__)
INTERNAL_CERT_PATH = '/usr/share/yandex-internal-root-ca/YandexInternalRootCA.crt'


class ConnectorBase(connectors.ConnectorBase):
    def handle_request(self, req_info):
        return self.vins_app.handle_request(req_info=req_info)

    def handle_response(self, chat_id, response):
        raise NotImplementedError()

    def bot_handler(self, bot, update):
        raise NotImplementedError()

    def run(self):
        raise NotImplementedError()

    def _setup_webhook(self, bot_token):
        if not os.environ.get('WEBHOOK'):
            logger.warning('Webhook is switched off in this environment')
            return

        domain = os.environ.get('WEBHOOK_DOMAIN')
        if not domain:
            logger.error('No WEBHOOK_DOMAIN environment variable, webhook is not installed')
            return

        url = 'https://%s/webhook/%s/' % (domain, hasher.sha256(bot_token))
        cert = None
        if yenv.type != 'production':
            cert = open(INTERNAL_CERT_PATH, 'rb')
        self.bot.set_webhook(url, cert)
        logger.info('Webhook url [%s] installed with cert [%r]', url, cert)

    def _list_to_keyboard(self, keyboard_list, buttons_in_row_count, cancel_button):
        keyboard = []
        cur_row = []
        i = 0
        for button in keyboard_list:
            if i < buttons_in_row_count:
                cur_row.append(button)
                i += 1
            else:
                keyboard.append(cur_row)
                cur_row = [button]
                i = 1
        if cur_row:
            keyboard.append(cur_row)
        if cancel_button:
            keyboard.append(['Отмена'])
        return keyboard

    def get_message_files(self, message, only_images=True):
        if message is None:
            return None

        if message.get('document'):
            document = message['document']
            mime_type = document.get('mime_type')
            if not only_images or (mime_type and (mime_type.split('/')[0] == 'image' or mime_type == 'video/mp4')):
                return [(document['file_name'], document['file_id'])]
        elif message.get('photo'):
            x = message['photo'][-1]
            return [(x.get('file_name'), x['file_id'])]

        return None

    def download_file(self, file_id, file_name):
        file_extension = self.get_extension_from_filename(file_name) or 'jpg'
        file_suffix = datetime.now().strftime('%Y-%m-%dT%H:%M:%S.%f')
        file_name = 'uhura_{}.{}'.format(file_suffix, file_extension)
        file = self.bot.get_file(file_id)
        file.download(file_name)
        return file_name

    def download_message_files(self, message, only_images=True):
        files = self.get_message_files(message, only_images)
        return [self.download_file(file_id, file_name) for (file_name, file_id) in files]

    def get_extension_from_filename(self, file_name):
        if not file_name or '.' not in file_name:
            return None
        else:
            return file_name.split('.')[-1]
