# coding: utf-8

from __future__ import unicode_literals

import functools
import json
import logging

from django.conf import settings
import telegram
from telegram import ext as telegram_ext
from telegram.utils import request as telegram_request
import waffle

from vins_core.dm.request import AppInfo, ReqInfo
from vins_core.common.utterance import Utterance
from vins_sdk import connectors as vins_connectors

from uhura.lib import cache_storage
from uhura.utils import strings
from .base_connector import ConnectorBase

logger = logging.getLogger(__name__)
TELEGRAM_API_RETRIES = 5


def _retry_on_timeout(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        last_exc = None
        retries = 1 if waffle.switch_is_active('not_use_telegram_retries') else TELEGRAM_API_RETRIES
        for _ in xrange(retries):
            try:
                return func(*args, **kwargs)
            except telegram.error.TimedOut as e:
                last_exc = e
        raise last_exc
    return wrapper


class TelegramConnector(ConnectorBase, vins_connectors.TelegramConnector):
    APP_INFO = AppInfo(**{'app_id': 'uhura_telegram'})
    COMMANDS_MAP = {
        '/start': '/start',
    }
    MAX_MESSAGE_LENGTH = 4096

    def __init__(self, bot_token, app, **kwargs):
        self.bot = telegram.Bot(
            settings.TELEGRAM_TOKEN,
            request=telegram_request.Request(con_pool_size=100, connect_timeout=5, read_timeout=5)
        )
        self._setup_webhook(settings.TELEGRAM_TOKEN)
        super(TelegramConnector, self).__init__(bot_token, self.COMMANDS_MAP, vins_app=app)
        self.vins_app.connector = self

    def handle_response(self, chat_id, response):
        if not response.vins_response_overrided:
            logger.debug('Old style handle vins response')
            self._handle_response(chat_id, response, parse_mode='HTML')
            return

        if response.images:
            for (img_str, img_type, cache_key, cache_value, cache_timeout) in response.images:
                if img_type == 'path':
                    file_id = self._send_file(chat_id, img_str, file_type='photo')
                    if cache_key:
                        cache_value['file_id'] = file_id
                        cache_storage.set(cache_key, cache_value, cache_timeout)

                elif img_type == 'file_id':
                    self._send_file(chat_id, file_type='photo', file_id=img_str)

                else:
                    raise ValueError('Invalid img type, not in [path, file_id]')

        inline_keyboards = list(reversed(response.inline_keyboards))

        if response.messages:
            if response.join_messages:
                messages = strings.join_with_max_length(response.messages, self.MAX_MESSAGE_LENGTH, '\n')
            else:
                messages = response.messages
            for message in messages[:-1]:
                if message:
                    if inline_keyboards:
                        self._send_message(
                            chat_id, message, disable_notification=True, inline_keyboard=inline_keyboards.pop()
                        )
                    else:
                        self._send_message(chat_id, message, disable_notification=True)
                else:
                    raise ValueError('Trying to send empty message')
            last_message = messages[-1]
        else:
            last_message = None

        keyboard = []
        if response.buttons:
            keyboard.extend(response.buttons)
        if response.request_user_contact:
            keyboard.extend(self._user_phone_keyboard())

        if last_message and (keyboard or response.cancel_button):
            self._send_keyboard(
                chat_id,
                keyboard,
                last_message,
                buttons_in_row_count=response.buttons_in_row_count,
                cancel_button=response.cancel_button
            )
        elif last_message and not keyboard:
            if inline_keyboards:
                self._send_message(chat_id, last_message, inline_keyboard=inline_keyboards.pop())
            else:
                self._send_message(chat_id, last_message)
        elif keyboard:
            raise ValueError('Trying to send keyboard without message %r', keyboard)

        if response.forward_messages:
            for forward_data in response.forward_messages:
                (to_chat_id, from_chat_id, message_id) = forward_data
                self._forward_message(to_chat_id, from_chat_id, message_id)
        if response.service_messages:
            for srv_data in response.service_messages:
                (to_chat_id, text) = srv_data
                self._send_message(to_chat_id, text)

    def error(self, bot, update, error):
        logger.error('Update "%s" caused error "%s"', update, error)

    def bot_handler(self, bot, update):
        logger.debug('Telegram update received: %r', update)
        try:
            if update.message:
                self.bot.sendChatAction(update.message.chat_id, telegram.ChatAction.TYPING)
        except telegram.TelegramError:
            logger.warning('Telegram sendChatAction failed')

        req_info = self._create_req_info(update)
        response = self.handle_request(req_info)
        self.handle_response(req_info.uuid, response)

    def run(self):
        self.updater = telegram_ext.Updater(bot=self.bot, workers=1)
        dispatcher = self.updater.dispatcher
        dispatcher.add_handler(telegram_ext.MessageHandler([lambda x: True], self.bot_handler))
        dispatcher.add_handler(telegram_ext.CallbackQueryHandler(self.bot_handler))
        dispatcher.add_error_handler(self.error)
        self.updater.start_polling()
        self.updater.idle()

    @_retry_on_timeout
    def download_file(self, file_id, file_name):
        return super(TelegramConnector, self).download_file(file_id, file_name)

    def _create_req_info(self, update):
        additional_options = {}
        if update.callback_query:
            callback_query = update.callback_query
            utterance = callback_query.data
            chat_id = callback_query.from_user.id
            client_time = callback_query.message.date
            additional_options['telegram_username'] = callback_query.from_user.username
            additional_options['change_to_empty'] = True
        else:
            message = update.message
            utterance = message.text or ''
            chat_id = message.chat_id
            client_time = message.date
            additional_options['telegram_username'] = message.from_user.username
            additional_options['forward_from'] = message.forward_from.to_dict() if message.forward_from else None
            additional_options['contact_info'] = message.contact.to_dict() if message.contact else None
            additional_options['message_id'] = message.message_id
            additional_options['message'] = message.to_dict()

        req_info = ReqInfo(
            uuid=self._to_uuid(chat_id),
            utterance=Utterance(utterance),
            client_time=client_time,
            app_info=self.APP_INFO,
        )
        req_info.additional_options.update(additional_options)
        return req_info

    def _send_file(self, chat_id, path=None, file_type='photo', file_id=None):
        possible_types = ['photo', 'audio', 'document', 'sticker', 'video', 'videonote']
        assert file_type in possible_types, 'Invalid file_type %s' % file_type
        tg_func = getattr(self.bot, 'send_' + file_type)
        kwargs = {'chat_id': chat_id}
        if path is not None:
            kwargs[file_type] = open(path.encode('utf-8'), 'rb')
        elif file_id is not None:
            kwargs[file_type] = file_id
        else:
            raise ValueError('path or file_id must be not None')
        message_obj = tg_func(**kwargs)
        file_obj = getattr(message_obj, file_type)
        if isinstance(file_obj, list):
            file_obj = file_obj[0]
        return file_obj.file_id

    @_retry_on_timeout
    def _send_keyboard(self, chat_id, keyboard, text='', parse_mode='HTML', buttons_in_row_count=2,
                       cancel_button=False):
        if not (keyboard or cancel_button):
            raise ValueError('no keyboard for send')
        reply_markup = json.dumps(dict(
            keyboard=self._list_to_keyboard(keyboard, buttons_in_row_count, cancel_button),
            resize_keyboard=True,
            one_time_keyboard=True,
        ))
        self.bot.send_message(chat_id, text, parse_mode, reply_markup=reply_markup)

    @_retry_on_timeout
    def _send_message(self, chat_id, text, parse_mode='HTML', disable_notification=False, inline_keyboard=None):
        if inline_keyboard:
            reply_markup = telegram.InlineKeyboardMarkup.de_json(inline_keyboard, self.bot)
        else:
            reply_markup = telegram.ReplyKeyboardRemove()
        self.bot.send_message(
            chat_id, text, parse_mode, reply_markup=reply_markup, disable_notification=disable_notification
        )

    def _user_phone_keyboard(self):
        contact_key = telegram.KeyboardButton(text='Отправить номер телефона', request_contact=True).to_dict()
        return [contact_key]

    def _forward_message(self, to_chat_id, from_chat_id, message_id):
        self.bot.forward_message(to_chat_id, from_chat_id, message_id=message_id)
