# coding: utf-8

from __future__ import unicode_literals

from datetime import datetime
import logging

from django.conf import settings

from vins_core.dm.request import AppInfo, ReqInfo
from vins_core.common.utterance import Utterance

from .base_connector import ConnectorBase
from uhura.lib import cache_storage
from uhura.lib.yamb import Bot, Updater
from uhura.utils import strings

logger = logging.getLogger(__name__)


class YambConnector(ConnectorBase):
    APP_INFO = AppInfo(**{'app_id': 'uhura_yamb'})

    def __init__(self, bot_token, app, **kwargs):
        self.bot = Bot(bot_token, base_url=settings.YAMB_API_BASE_URL)
        app.bot = self.bot
        app.connector = self
        self._setup_webhook(bot_token)
        super(YambConnector, self).__init__(vins_app=app, **kwargs)

    def _send_message(self, chat_id, text, keyboard):
        logger.info('Send message: %s', text)
        self.bot.send_message(chat_id=chat_id, text=text, reply_markup=keyboard)

    def _send_file(self, chat_id, path=None, file_type='photo', file_id=None):
        possible_types = ['photo']
        assert file_type in possible_types, 'Invalid file_type %s' % file_type
        tg_func = getattr(self.bot, 'send_' + file_type)
        kwargs = {'chat_id': chat_id}
        if path is not None:
            kwargs[file_type] = open(path.encode('utf-8'), 'rb')
        elif file_id is not None:
            kwargs['file_id'] = file_id
        else:
            raise ValueError('path or file_id must be not None')
        message_obj = tg_func(**kwargs)
        file_obj = message_obj[file_type]
        if isinstance(file_obj, list):
            file_obj = file_obj[0]
        return file_obj['file_id']

    def handle_response(self, chat_id, response):
        if not response.vins_response_overrided:
            for card in response.cards:
                self.bot.send_message(chat_id=chat_id, text=card.text)
            return

        if response.images or response.forward_messages or response.request_user_contact or response.service_messages:
            for (img_str, img_type, cache_key, cache_value, cache_timeout) in response.images:
                if img_type == 'path':
                    file_id = self._send_file(chat_id, img_str, file_type='photo')
                    if cache_key:
                        cache_value['file_id'] = file_id
                        cache_storage.set(cache_key, cache_value, cache_timeout)

                elif img_type == 'file_id':
                    self._send_file(chat_id, file_type='photo', file_id=img_str)

                else:
                    raise ValueError('Invalid img type, not in [path, file_id]')

        for message in response.messages[:-1]:
            message = strings.unescape_html(message)
            self._send_message(chat_id, message, None)

        reply_markup = None
        if response.buttons or response.cancel_button:
            keyboard = self._list_to_keyboard(response.buttons, response.buttons_in_row_count, response.cancel_button)
            reply_markup = {'inline_keyboard': [
                [{'text': key} for key in row]
                for row in keyboard
            ]}

        self._send_message(chat_id, response.messages[-1], reply_markup)

    def bot_handler(self, bot, update):
        logger.debug('Yamb update received: %r', update)
        message = update['message']
        utterance = message.get('text') or ''
        uuid = message['chat']['id']
        req_info = ReqInfo(
            uuid=uuid,
            utterance=Utterance(utterance),
            client_time=datetime.now(),
            app_info=self.APP_INFO,
        )

        req_info.additional_options['uid'] = message['from']['uid']
        req_info.additional_options['message'] = {
            'text': utterance,
            'from': {'id': uuid},
            'photo': message.get('photo'),
            'document': message.get('document'),
        }
        response = self.handle_request(req_info)
        self.handle_response(uuid, response)

    def run(self):
        updater = Updater(self.bot)
        for (update, error) in updater.poll():
            try:
                self.bot_handler(self.bot, update)
            except Exception:
                logger.exception('Update %s caused error', update)
