# coding: utf-8

import logging
import operator

logger = logging.getLogger(__name__)

DEFAULT_CHAIN_BOOST = 10


class BaseRule(object):
    def __init__(self, boost, *args, **kwargs):
        self._boost = boost

    @property
    def boost(self):
        if isinstance(self._boost, (int, float)):
            return self._boost
        else:
            raise ValueError("Boost must be specified for top level rules")

    def check(self, *args, **kwargs):
        raise NotImplementedError


class CheckPrevCurrIntentRule(BaseRule):
    def __init__(self, prev_intent, curr_intent, *args, **kwargs):
        super(CheckPrevCurrIntentRule, self).__init__(*args, **kwargs)
        self._prev_intent = prev_intent
        self._curr_intent = curr_intent

    def check(self, intent_name, session, req_info=None):
        '''
        compare current and previous intent in transition with intents specified in rule,
        if compare success transition will be boosted
        :param intent_name: name of the intent on which can go
        :param session: current user session
        :param req_info: request info
        '''
        if self._prev_intent != session.intent_name or self._curr_intent != intent_name:
            return False
        return True


class CheckFormActiveSlots(BaseRule):
    def __init__(self, prev_intent, curr_intent, *args, **kwargs):
        super(CheckFormActiveSlots, self).__init__(*args, **kwargs)
        self._prev_intent = prev_intent
        self._curr_intent = curr_intent

    def check(self, intent_name, session, req_info=None):
        '''
        if form have active slots boost transition prev_intent -> curr_intent
        :param intent_name: name of the intent on which can go
        :param session: current user session
        :param req_info: request info
        '''
        if session.form and session.form.has_active_slots():
            return self._prev_intent == session.intent_name and self._curr_intent == intent_name
        return False


class CheckFormSlotValueRule(BaseRule):
    def __init__(self, slot, slot_value_key, value, *args, **kwargs):
        super(CheckFormSlotValueRule, self).__init__(*args, **kwargs)
        self._slot = slot
        self._slot_value_key = slot_value_key
        self._value = value

    def check(self, intent_name, session, req_info=None):
        '''
        check conditions of the rule are satisfied
        will find special slot in current form and then compare it\'s value
        :param intent_name: name of the intent on which can go
        :param session: current user session
        :param req_info: request info
        '''
        if not session.form:
            return False
        form = session.form
        form_slot = form.get_slot_by_name(self._slot)
        if not form_slot:
            return False
        val = form_slot.value
        if not isinstance(val, dict):
            return False
        return val and val.get(self._slot_value_key) == self._value


class ChainRule(BaseRule):
    def __init__(self, intents_chain, module, *args, **kwargs):
        super(ChainRule, self).__init__(*args, **kwargs)
        self._chain = ['uhura.{module}.{name}'.format(module=module, name=name) for name in intents_chain]

    def check(self, intent_name, session, req_info=None):
        prev_intent_name = session.intent_name
        try:
            prev_intent_index = self._chain.index(prev_intent_name)
            cur_intent_index = self._chain.index(intent_name)
            return cur_intent_index - prev_intent_index == 1  # return True if cur_intent is next
        except ValueError:  # prev_intent or current_intent is not in chain
            return False


_operations = {
    'and': operator.and_,
    'or': operator.or_,
}


class LogicRule(BaseRule):
    def __init__(self, operation, children, *args, **kwargs):
        super(LogicRule, self).__init__(*args, **kwargs)
        self._op = _operations.get(operation)
        self._op_str = operation
        if not self._op:
            raise ValueError("Can't find logic rule operand")
        self._children = []
        for child in children:
            self._children.extend(create_transition_rules([child]))
        if len(self._children) < 2:
            raise ValueError("Logic rule must have two or more children rules")

    def check(self, intent_name, session, req_info=None):
        '''
        logic rule checks result of or/and operation over all
        children rules
        :param intent_name: name of the intent on which can go
        :param session: current user session
        :param req_info: request info
        '''
        res = self._children[0].check(intent_name, session, req_info)
        for child in self._children[1:]:
            # We can finish calculating the rule in advance
            if res and self._op_str == 'and' or not res and self._op_str == 'or':
                res = self._op(res, child.check(intent_name, session, req_info))
            else:
                return res
        return res


def create_transition_rules(rules):
    all_rules = []
    for rule in rules:
        try:
            if rule['type'] == 'check_form_slot_value':
                all_rules.append(CheckFormSlotValueRule(
                    rule['slot'], rule['slot_value_key'], rule['value'],
                    boost=rule.get('boost'))
                )
            elif rule['type'] == 'check_prev_curr_intent':
                all_rules.append(CheckPrevCurrIntentRule(rule['prev_intent'], rule['curr_intent'],
                                 boost=rule.get('boost')))
            elif rule['type'] == 'logic_rule':
                all_rules.append(LogicRule(rule['operation'], rule['children'],
                                 boost=rule.get('boost')))
            elif rule['type'] == 'check_form_active_slots':
                all_rules.append(CheckFormActiveSlots(rule['prev_intent'], rule['curr_intent'],
                                 boost=rule.get('boost')))
            elif rule['type'] == 'chain_rule':
                all_rules.append(ChainRule(rule['chain'], rule['module'], boost=rule.get('boost', DEFAULT_CHAIN_BOOST)))
            else:
                raise ValueError("Unknown rule type")
        except Exception as e:
            logger.error("Can't create rule, skip it: {} error: {}".format(str(rule), str(e)))
    return all_rules
