# coding: utf-8

from __future__ import unicode_literals

import logging
import urlparse

from django.conf import settings

from uhura.external import intranet

logger = logging.getLogger(__name__)


class WebhookError(Exception):
    pass


class Bot(object):
    def __init__(self, token, base_url):
        self.base_url = base_url
        self.token = token
        self.auth_headers = {'Authorization': 'OAuthTeam %s' % token}

    def _get(self, url,  *args, **kwargs):
        return intranet.get_request(url, *args, auth_headers=self.auth_headers, **kwargs)

    def _post(self, url,  *args, **kwargs):
        return intranet.post_request(url, *args, auth_headers=self.auth_headers, **kwargs)

    def get_url(self, method):
        return urlparse.urljoin(self.base_url, method)

    def get_updates(self, offset=None):
        return self._get(self.get_url('getUpdates'), params={'offset': offset})

    def send_message(self, chat_id, text, reply_markup=None):
        return self._post(
            self.get_url('sendMessage'),
            data={'chat_id': chat_id, 'text': text, 'reply_markup': reply_markup}
        )

    def send_photo(self, chat_id, file_id=None, photo=None, width=None, height=None, size=None):
        if (file_id and photo) or (not file_id and not photo):
            raise ValueError('One and only one of (file_id, file) should be provided')

        if file_id is not None:
            assert width is not None and height is not None and size is not None  # обязательные параметры для Q
            data = {
                'file_id': file_id, 'chat_id': chat_id, 'filename': 'image.png',
                'width': width, 'height': height, 'size': size,
            }
            result = self._post(self.get_url('sendPhoto'), data=data)
        elif photo is not None:
            result = self._post(self.get_url('sendPhoto'), data={'chat_id': chat_id}, files={'photo': photo})
        else:
            raise ValueError('file_id or photo should be specified')
        return result['message']

    def set_webhook(self, webhook_url, cert):
        response = self._post(settings.YAMB_UPDATE_BOT_URL, data={'webhook_url': webhook_url})
        if response is None:
            raise WebhookError()

    def get_file(self, file_id):
        return File(self, file_id)

    def download_file(self, file_id):
        return self._get(settings.YAMB_API_FILE_URL % file_id)


class File(object):
    def __init__(self, bot, file_id):
        self.bot = bot
        self.file_id = file_id

    def download(self, file_name):
        buf = self.bot._get(settings.YAMB_API_FILE_URL % self.file_id, raise_exception=True, raw_response=True)
        with open(file_name, 'wb') as f:
            f.write(buf)


class Update(dict):
    @property
    def update_id(self):
        return self['message_id']

    @classmethod
    def de_json(cls, data, bot):
        return cls(data)


class Updater(object):
    def __init__(self, bot):
        self.bot = bot

    def poll(self):
        update = None
        offset = None
        while True:
            try:
                for update in self.bot.get_updates(offset=offset):
                    yield update, None
                if update:
                    offset = update['update_id'] + 1
            except Exception:
                logger.exception('Exception during polling')
