# coding: utf-8

from datetime import date, timedelta
import logging

from django.core.management.base import BaseCommand

from uhura.management.commands.utils import start_query, table_exists


QUERY_TEMPLATE = '''
$script = @@
import cyson as yson
import json
from yql.typing import *

def process_message(
        message: Optional[String],
        qloud_application: Optional[String],
        qloud_environment: Optional[String],
    ) -> Optional[Struct[
            'app_id':String,
            'client_time':Uint64,
            'request':Optional[Yson],
            'response':Optional[Yson],
            'form':Optional[Yson],
            'form_name':String,
            'provider':String,
            'type':String,
            'server_time':Uint64,
            'utterance_source':String,
            'utterance_text':Optional[String],
            'uuid':String,
            'callback_name':Optional[String],
            'callback_args':Optional[Yson],
            'location_lat':Optional[Double],
            'location_lon':Optional[Double],
            'lang':String,
            'experiments':Optional[Yson],
            'device_id':Optional[String],
            'environment':String,
        ]]:
    try:
        dialog_data = json.loads(message)
    except ValueError:
        return None

    dialog_data['uuid'] = 'uu/' + dialog_data['uuid']

    # coerce lat/lon values to float because table schema requires this
    lat = dialog_data['location_lat']
    lon = dialog_data['location_lon']
    dialog_data['location_lat'] = float(lat) if lat is not None else None
    dialog_data['location_lon'] = float(lon) if lon is not None else None

    dialog_data['environment'] = '_'.join([
        qloud_application.decode('utf-8'), 
        qloud_environment.decode('utf-8'),
    ])
    for key in dialog_data:
        if isinstance(dialog_data[key], dict) or isinstance(dialog_data[key], list):
            dialog_data[key] = yson.dumps(dialog_data[key])
    return dialog_data
@@;

$f = Python::process_message($script);

USE hahn;
INSERT INTO `home/uhura/dialog_history/{messenger}/{env_type}/{uhura_table_name}` WITH TRUNCATE
PROCESS(
    SELECT *
    FROM `home/logfeller/logs/qloud-runtime-log/1d/{qloud_table_name}`
    WHERE
        qloud_project     = "tools" AND
        qloud_application = "uhura" AND
        qloud_environment = "{qloud_environment}" AND
        loggerName        = "syslog"
) USING $f(message, qloud_application, qloud_environment)
'''
ENVS_LIST = ['telegram-production', 'yamb-production']
MAX_RETRIES = 5


logger = logging.getLogger(__name__)


class Command(BaseCommand):
    help = 'Copy dialog logs from qloud to hanh/home/uhura'

    def handle(self, *args, **options):
        prev_date = date.today() - timedelta(days=2)
        while not (
            table_exists('home/uhura/dialog_history/telegram/production/{}'.format(prev_date.strftime('%Y-%m-%d')))
            and table_exists('home/uhura/dialog_history/yamb/production/{}'.format(prev_date.strftime('%Y-%m-%d')))
        ):
            prev_date -= timedelta(days=1)
        prev_date += timedelta(days=1)
        while prev_date < date.today():
            for env in ENVS_LIST:
                table_name = prev_date.strftime('%Y-%m-%d')
                format_dict = {
                    'env_type': env.split('-')[1],
                    'messenger': env.split('-')[0],
                    'uhura_table_name': table_name,
                    'qloud_table_name': table_name,
                    'qloud_environment': env
                }
                query_text = QUERY_TEMPLATE.format(**format_dict)
                start_query(query_text)
            prev_date += timedelta(days=1)
