# coding: utf-8

from datetime import date, timedelta
import logging
import os

from django.core.management.base import BaseCommand

from uhura.external.startrek import create_ticket
from uhura.management.commands.utils import start_query

from yql.api.v1.client import YqlClient


QUERY_TEMPLATE = '''
USE hahn;

SELECT
    DateTime::ToString(DateTime::FromSeconds(client_time)) as datetime,
    utterance_text
FROM RANGE(
    [home/uhura/dialog_history/{messenger}/{env_type}],
    "{from_date}", "{to_date}"
)
WHERE type = "UTTERANCE" and form_name = "{form_name}"
ORDER by datetime;
'''
REQUIRED_FORMS = ['dont_understand', 'feedback__message']
ENVS_LIST = ['telegram-production', 'yamb-production']
MAX_RETRIES = 5


logger = logging.getLogger(__name__)
yql_client = YqlClient()


def _save_logs_to_tsv(query_results, filename):
    for table in query_results:
        table.fetch_full_data()
        with open(filename, 'w') as f:
            for row in table.rows:
                f.write(('\t'.join([unicode(cell) for cell in row]) + '\n').encode('utf-8'))


class Command(BaseCommand):
    help = 'Create ticket with feedback'

    def handle(self, *args, **options):
        cur_date = date.today()
        prev_monday = cur_date - timedelta(days=(7 + cur_date.weekday()))
        prev_sunday = prev_monday + timedelta(days=6)
        format_dict = {
            'from_date': prev_monday.strftime('%Y-%m-%d'),
            'to_date': prev_sunday.strftime('%Y-%m-%d')
        }

        files = []
        for env in ENVS_LIST:
            format_dict.update({
                'env_type': env.split('-')[1],
                'messenger': env.split('-')[0]
            })
            for form in REQUIRED_FORMS:
                format_dict['form_name'] = 'uhura.utils.{}'.format(form)
                query_text = QUERY_TEMPLATE.format(**format_dict)
                filename = '{form}_{messenger}.tsv'.format(form=form, **format_dict)
                logs = start_query(query_text).get_results()
                _save_logs_to_tsv(logs, filename)
                files.append(filename)
        if files:
            ticket_summary = 'Фидбек с {} по {}'.format(prev_monday.strftime('%d %B'), prev_sunday.strftime('%d %B'))
            create_ticket(
                login='robot-uhura',
                queue='uhura',
                summary=ticket_summary,
                attachments=filter(lambda x: os.stat(x).st_size, files),
                tags=['Uhura_feedback']
            )
            for file_name in files:
                os.remove(file_name)
