# coding: utf-8

from __future__ import unicode_literals

import codecs
import collections
import json
import os
import re
import traceback

from django.conf import settings
from django.core.management.base import BaseCommand
import pyaml

from uhura.external import wiki


NONALPHANUM_RE = re.compile('\W+', re.UNICODE)
NONALPHANUM_PREFIX_RE = re.compile('^\W+(?={%)', re.UNICODE)
NLG_START_RE = re.compile('-\s+', re.UNICODE)


def _make_nlu(raw_nlu):
    raw_nlu = raw_nlu.replace("'", '')
    nlu_parts = raw_nlu.split('\n')
    prepared_parts = []
    for part in nlu_parts:
        part = part.lower()
        cleaned_part = re.sub(NONALPHANUM_RE, ' ', part).strip()
        if not cleaned_part:
            continue
        prepared_parts.append(cleaned_part)
    return prepared_parts


def _make_nlg(raw_nlg):
    raw_nlg = raw_nlg.replace("'", '')
    nlg_parts = NLG_START_RE.split(raw_nlg)
    prepared_parts = []
    for part in nlg_parts:
        cleaned_part = re.sub(NONALPHANUM_PREFIX_RE, '', part).strip()
        if not cleaned_part:
            continue
        cleaned_part = cleaned_part.replace('\n\n', '\n').replace('\n', '\\n')
        prepared_parts.append(cleaned_part)
    return prepared_parts


def update_config(talk):
    microintents_path = 'microintents/%s.yaml' % talk
    with codecs.open(settings.TALKS_CONFIG_PATH, encoding='utf8') as fin:
        config_data = pyaml.yaml.load(fin)
    all_microintets = set(m['path'] for m in config_data['microintents'])
    if microintents_path in all_microintets:
        return

    config_data['microintents'].append({
        'path': microintents_path,
        'fallback_threshold': 0.93,
        'trainable_classifiers': ['uhura'],
    })
    with codecs.open(settings.TALKS_CONFIG_PATH, 'w', encoding='utf8') as fout:
        pyaml.yaml.dump(config_data, fout, indent=2)
        fout.write('\n')


class Command(BaseCommand):
    help = 'Get Uhura\'s talks from wiki'

    def add_arguments(self, parser):
        parser.add_argument('talk_id', nargs='+', type=str)

    def handle(self, *args, **options):
        try:
            os.mkdir(settings.TALKS_MICROINTENTS_PATH)
            self.stdout.write(self.style.SUCCESS(
                'Path created %s' % settings.TALKS_MICROINTENTS_PATH
            ))
        except OSError:
            self.stdout.write(self.style.SUCCESS(
                'Path already exists %s' % settings.TALKS_MICROINTENTS_PATH
            ))

        talks = list(options['talk_id'])
        grids_for_handle = {}
        if not talks or (len(talks) == 1 and talks[0] == 'all'):
            grids_for_handle = settings.WIKI_TALKS_GRIDS
        else:
            for t in talks:
                if t in settings.WIKI_TALKS_GRIDS:
                    grids_for_handle[t] = settings.WIKI_TALKS_GRIDS[t]

        for (talk_file, grid_id) in grids_for_handle.items():
            grid_data = wiki.get_wiki_grid(grid_id)
            if grid_data is None:
                self.stdout.write(self.style.SUCCESS(
                    'Empty grid data for "%s" talks, no changes will be commited' % talk_file
                ))
                continue

            microintents = collections.OrderedDict()
            for row in grid_data['rows']:
                try:
                    intent_name = '%s_%s' % (talk_file, row[0]['raw'])
                    d = collections.OrderedDict()
                    d['nlu'] = _make_nlu(row[1]['raw'].strip())
                    d['nlg'] = _make_nlg(row[2]['raw'].strip())
                    microintents[intent_name] = d
                except Exception:
                    self.stderr.write(self.style.ERROR('Exception in row [%r]' % row))
                    self.stderr.write(self.style.ERROR(traceback.format_exc()))

            # if someone ocasionnally erases all wiki grid content
            if not microintents:
                continue

            talk_file_path = os.path.join(settings.TALKS_MICROINTENTS_PATH, talk_file + '.yaml')
            with open(talk_file_path, 'w') as fd:
                pyaml.dump(
                    microintents,
                    fd,
                    width=100,
                    indent=2,
                    canonical=False,
                    vspacing=[1, 0],
                )
            update_config(talk_file)
