# coding: utf-8
from __future__ import absolute_import, unicode_literals

import logging
import re

from django.conf import settings
from django.core.management.base import BaseCommand
from django.db import transaction
from django.utils import timezone

from uhura.external import staff
from uhura.lib.lock import locked
from uhura.models import User, SyncTime
from uhura.management.commands.utils import update_model

logger = logging.getLogger(__name__)


class Command(BaseCommand):
    def handle(self, *args, **options):
        with locked('import_staff') as lock:
            if not lock:
                return

            self._handle(*args, **options)

    def _handle(self, *args, **options):
        lookup = {
            '_fields': 'id,uid,login,phones,accounts,official.is_dismissed,work_email,official.quit_at',
            '_sort': 'id',
            '_limit': settings.STAFF_PAGE_LIMIT,
        }
        start_time = timezone.now() - timezone.timedelta(minutes=3)
        query_template = 'id > %d'
        last_time = None

        try:
            sync_time = SyncTime.objects.get(name='import_staff')
        except SyncTime.DoesNotExist:
            sync_time = SyncTime(name='import_staff')
        else:
            last_time = sync_time.last_success_start

        if last_time:
            query_template += ' and _meta.modified_at > "%s"' % sync_time.last_success_start.strftime('%Y-%m-%dT%H:%M')

        last_id = 0
        good_phone_re = re.compile(r'^\+?\d+$')
        phone_symbols_re = re.compile(r'[+\d]')
        while True:
            with transaction.atomic():
                lookup['_query'] = query_template % last_id
                logger.info('Import users with lookup %s', lookup)
                persons = list(staff.persons.getiter(lookup=lookup).first_page)
                if len(persons) == 0:
                    break
                last_id = persons[-1]['id']
                uids_to_persons = {int(person['uid']): person for person in persons}
                uids_to_users = {int(user.uid): user for user in User.objects.filter(uid__in=uids_to_persons.keys())}
                for uid, person in uids_to_persons.items():
                    phone = staff.find_main_phone(person['phones']) or ''
                    if phone and not good_phone_re.match(phone):
                        phone = ''.join(phone_symbols_re.findall(phone))
                        if not good_phone_re.match(phone):
                            phone = ''
                    fields = {
                        'username': person['login'],
                        'email': person['work_email'],
                        'is_active': not person['official']['is_dismissed'],
                        'phone': phone,
                        'quit_at': person['official']['quit_at'],
                    }
                    if uid in uids_to_users:
                        user = uids_to_users[uid]
                        if user.is_active and not fields['is_active']:
                            fields['leave_at'] = timezone.now()
                        update_model(user, fields)
                    else:
                        user = User.objects.create(uid=uid, **fields)
                    user.update_usernames(staff.find_telegram_accounts(person['accounts']))

        sync_time.last_success_start = start_time
        sync_time.save()
