# coding: utf-8
from __future__ import unicode_literals

import datetime
import urlparse

from django.conf import settings
from django.contrib.postgres import fields
from django.db import models
from requests import RequestException

from uhura.external.intranet import post_request
from uhura.utils.tvm import get_tvm_service_headers


class Session(models.Model):
    class Meta:
        app_label = 'uhura'
        unique_together = (('app_id', 'uuid'),)

    app_id = models.CharField(max_length=255)
    uuid = models.CharField(max_length=255)
    session = fields.JSONField(null=True)
    updated_at = models.DateTimeField(auto_now=True)


class SyncTime(models.Model):
    name = models.TextField()
    last_success_start = models.DateTimeField()


class User(models.Model):
    class Meta:
        app_label = 'uhura'

    def __unicode__(self):
        return self.uid

    uid = models.CharField(max_length=255, primary_key=True)
    username = models.TextField(unique=True, null=True)
    can_use_uhura = models.BooleanField(default=False)
    email = models.TextField(null=True)
    is_active = models.BooleanField(null=False, default=False)
    yamb_id = models.CharField(max_length=255, blank=True, default='')
    phone = models.CharField(max_length=255, blank=True, default='')
    city = models.CharField(max_length=255, blank=True, default='')
    quit_at = models.DateField(null=True)
    leave_at = models.DateTimeField(null=True)
    blocked_cars_plates = fields.ArrayField(models.CharField(max_length=255), blank=True, default=list)

    def update_usernames(self, telegram_usernames):
        new_usernames = {username.lower() for username in telegram_usernames}
        old_usernames = set(
            self.telegram_usernames
            .filter(username__isnull=False, created_with_uhura=False)
            .values_list('username', flat=True)
        )

        TelegramUsername.objects.filter(username__in=(old_usernames - new_usernames)).update(user=None)

        for username in new_usernames - old_usernames:
            try:
                telegram_username = TelegramUsername.objects.get(username=username)
            except TelegramUsername.DoesNotExist:
                TelegramUsername.objects.create(user=self, username=username, telegram_id=None)
            else:
                if telegram_username.user and not telegram_username.user.is_active:
                    # Телега привязанная к уволенному сотруднику на нем и остается
                    continue

                telegram_username.user = self
                telegram_username.save(update_fields=['user', 'telegram_id'])

    def get_usernames(self):
        return [tglogin.username for tglogin in self.telegram_usernames.all()]

    def register_telegram_id(self, telegram_id):
        telegram_username = TelegramUsername.objects.filter(telegram_id=telegram_id).first()
        if telegram_username:
            telegram_username.created_with_uhura = True
            telegram_username.username = None
            telegram_username.user = self
            telegram_username.pushed_to_tasha = False
            telegram_username.save(update_fields=['user', 'username', 'created_with_uhura', 'pushed_to_tasha'])
        else:
            self.telegram_usernames.create(
                telegram_id=telegram_id,
                created_with_uhura=True,
                pushed_to_tasha=False,
            )


class TelegramUsername(models.Model):
    class Meta:
        app_label = 'uhura'

    created_with_uhura = models.BooleanField(default=False)
    is_bot = models.BooleanField(default=False)
    is_tasha = models.BooleanField(default=False)
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='telegram_usernames', null=True)
    username = models.CharField(max_length=255, blank=True, null=True, db_index=True, unique=True)
    telegram_id = models.BigIntegerField(db_index=True, null=True, unique=True)
    pushed_to_tasha = models.BooleanField(default=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)

    def push_to_tasha(self):
        if self.pushed_to_tasha:
            return

        data = {'user': self.user.username, 'telegram_id': self.telegram_id}
        if post_request(
                urlparse.urljoin(settings.TASHA_API_HOST, settings.TASHA_REGISTRATION_PATH),
                data=data,
                raw_response=True,
                headers=get_tvm_service_headers(settings.TVM2_TASHA_ID),
            ) is not None:
            self.pushed_to_tasha = True
            self.save(update_fields=['pushed_to_tasha'])


class QuestProgress(models.Model):
    class Meta:
        app_label = 'uhura'

    user = models.OneToOneField(User, on_delete=models.CASCADE, primary_key=True)
    passed_tasks_count = models.IntegerField(blank=True, default=0)


class Subscription(models.Model):
    class Meta:
        app_label = 'uhura'

    name = models.CharField(max_length=255, unique=True)
    task_name = models.CharField(max_length=255, unique=True)
    description = models.CharField(max_length=1000)
    description_template = models.CharField(max_length=1000)


class PeriodicNotification(models.Model):
    class Meta:
        app_label = 'uhura'

    user = models.ForeignKey(User, on_delete=models.CASCADE)
    subscription = models.ForeignKey(Subscription, on_delete=models.CASCADE)
    time = models.TimeField()
    last_success_date = models.DateField(default=datetime.date(1970, 1, 1))
    params = fields.JSONField(blank=True, default=dict)


class EmergencyNotifier(models.Model):
    class Meta:
        app_label = 'uhura'

    def __unicode__(self):
        return self.staff_login

    staff_login = models.CharField(max_length=255, unique=True)


class EmergencyNotification(models.Model):
    class Meta:
        app_label = 'uhura'

    def __unicode__(self):
        return '{} {} {}'.format(self.answer_id, self.author, self.subject)

    answer_id = models.CharField(max_length=255, primary_key=True)
    author = models.CharField(max_length=255)
    subject = models.CharField(max_length=255)
    long_template = models.CharField(max_length=1000)
    short_template = models.CharField(max_length=255)
    mail_attempts = models.IntegerField(blank=True, default=0)
    mail_success = models.IntegerField(blank=True, default=0)
    telegram_attempts = models.IntegerField(blank=True, default=0)
    telegram_success = models.IntegerField(blank=True, default=0)
    yamb_attempts = models.IntegerField(blank=True, default=0)
    yamb_success = models.IntegerField(blank=True, default=0)
    sms_attempts = models.IntegerField(blank=True, default=0)
    sms_success = models.IntegerField(blank=True, default=0)


class EmergencyTestingList(models.Model):
    class Meta:
        app_label = 'uhura'

    def __unicode__(self):
        return self.login

    login = models.CharField(max_length=255, primary_key=True)


class Office(models.Model):
    class Meta:
        app_label = 'uhura'

    def __unicode__(self):
        return self.name

    name = models.CharField(max_length=255, unique=True)
    country_code = models.CharField(max_length=255, db_index=True)
    city_code = models.CharField(max_length=255, db_index=True)
    office_code = models.CharField(max_length=255, primary_key=True)
    floor_codes = fields.ArrayField(models.CharField(max_length=255))
    emergency_emails = fields.ArrayField(models.CharField(max_length=255))


class UserToken(models.Model):
    class Meta:
        app_label = 'uhura'

    user = models.OneToOneField(User)
    token = models.CharField(max_length=39)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
