# coding: utf-8

from __future__ import unicode_literals

import collections
import logging
import re
import sys
import traceback
import urllib
import xml

import celery
from django.conf import settings
from django.db import models as django_models
from django.db.utils import DatabaseError
import jinja2
import requests
import telegram

from ids.utils import https as ids_https
from ids import exceptions as ids_exceptions
import yenv

from uhura import models
from uhura.external import staff
from uhura.lib import cache_storage
from uhura.lib import mail
from uhura.lib.yamb import Bot


COUNTRY_QUERY = 'location.office.city.country.id == {}'
CITY_QUERY = 'location.office.city.id == {}'
OFFICE_QUERY = 'location.office.id == {}'
FLOOR_QUERY = 'location.table.floor.id == {}'
STAFF_ID_RANGE = 1000
MAX_RETRIES = 5
TEMPLATE_CACHE_TIME = 1200  # seconds

SMS_TIMEOUT = 0.5

AURORA_OFFICE_ID = 181
AURORA_SECTION_NAME_REGEXP = re.compile(r'\w* (\w)-\w*', re.UNICODE)  # get "A" from "Sixth A-floor"
AURORA_SECTION_FLOORS_MAPPING = {
    'A': ['96', '98', '99', '103', '114', '132', '154'],
    'B': ['102', '104', '116', '127'],
    'C': ['153', '155', '156', '157', '170'],
    'F': ['105', '117', '122', '165']
}

jinja_env = jinja2.Environment()
logger = logging.getLogger(__name__)
verify = ids_https.get_ca_bundle_path() or False
telegram_bot = telegram.Bot(settings.TELEGRAM_TOKEN)
yamb_bot = Bot(settings.YAMB_TOKEN, base_url=settings.YAMB_API_BASE_URL)


@cache_storage.memoize_decorator(TEMPLATE_CACHE_TIME)
def get_notification_templates(answer_id):
    notification_data = models.EmergencyNotification.objects.get(answer_id=answer_id)
    subject = notification_data.subject
    short_template = notification_data.short_template
    long_template = notification_data.long_template
    return {
        'subject': subject,
        'short_template': short_template,
        'long_template': long_template,
    }


@cache_storage.memoize_decorator(TEMPLATE_CACHE_TIME)
def render_templates_by_answer_id(answer_id, context):
    data = get_notification_templates(answer_id)
    (subject, short_template, long_template) = (data['subject'], data['short_template'], data['long_template'])
    subject = jinja_env.from_string(subject).render(**context)
    short = jinja_env.from_string(short_template).render(**context)
    long_ = jinja_env.from_string(long_template).render(**context)
    return {
        'subject': subject,
        'short': short,
        'long': long_,
    }


def increase_field(answer_id, field_name, value=1):
    pass
    # try:
    #     update_data = {field_name: django_models.F(field_name) + value}
    #     models.EmergencyNotification.objects.filter(answer_id=answer_id).update(**update_data)
    # except Exception:
    #     logger.exception('Increasing answer_id %s field %s raised exception', answer_id, field_name)


@cache_storage.memoize_decorator(TEMPLATE_CACHE_TIME)
def get_testing_logins(test_run):
    testing_logins = set()
    if test_run:
        testing_logins = set(models.EmergencyTestingList.objects.values_list('login', flat=True))
    return testing_logins


@celery.shared_task(autoretry_for=(Exception,), retry_kwargs={'max_retries': MAX_RETRIES, 'countdown': 2})
def telegram_notify(chat_id, answer_id, context, dont_send=False):
    logger.info('Start sending telegram_message to %s for answer_id %s', chat_id, answer_id)
    text = render_templates_by_answer_id(answer_id, context)['short']
    if dont_send:
        logger.info('Do not send telegram notification to %s in %s env', chat_id, yenv.type.upper())
    else:
        telegram_bot.send_message(chat_id=chat_id, text=text)
        logger.info('Successfully sent telegram_message to %s for answer_id %s', chat_id, answer_id)
        increase_field(answer_id, 'telegram_success')


@celery.shared_task(autoretry_for=(Exception,), retry_kwargs={'max_retries': MAX_RETRIES, 'countdown': 2})
def q_notify(chat_id, answer_id, context, dont_send=False):
    logger.info('Start sending q_message to %s for answer_id %s', chat_id, answer_id)
    text = render_templates_by_answer_id(answer_id, context)['short']
    if dont_send:
        logger.info('Do not send q_message to %s in %s env', chat_id, yenv.type.upper())
    else:
        yamb_bot.send_message(chat_id=chat_id, text=text)
        logger.info('Successfully sent q_message to %s for answer_id %s', chat_id, answer_id)
        increase_field(answer_id, 'yamb_success')


@celery.shared_task(autoretry_for=(Exception,), retry_kwargs={'max_retries': MAX_RETRIES, 'countdown': 2})
def mail_notify(to, answer_id, context, dont_send=False, use_bcc=False):
    logger.info('Start sending email to %s for answer_id %s', to, answer_id)
    data = render_templates_by_answer_id(answer_id, context)
    (subject, text) = (data['subject'], data['long'])
    if dont_send:
        logger.info('Do not send mail notification to %s in %s env', to, yenv.type.upper())
    else:
        mail.send_mail(settings.UHURA_MAIL, to, subject, text=text, use_bcc=use_bcc)
        logger.info('Successfully sent email to %s for answer_id %s', to, answer_id)
        increase_field(answer_id, 'mail_success')


@celery.shared_task(autoretry_for=(Exception,), retry_kwargs={'max_retries': MAX_RETRIES, 'countdown': 2})
def sms_notify(phone, answer_id, context, dont_send=False):
    logger.info('Start sending sms to %s for answer_id %s', phone, answer_id)
    text = render_templates_by_answer_id(answer_id, context)['short']
    if dont_send:
        logger.info('Do not send sms notification to %s in %s env', phone, yenv.type.upper())
    else:
        if yenv.type != 'production':
            text = '[{}]'.format(yenv.type.upper()) + text
        text = text.replace('ОТ ЯНДЕКСА: ', '')

        url = '{host}/sendsms?{params}'.format(
            host=settings.SMS_PASSPORT_HOST,
            params=urllib.urlencode({'sender': 'uhura', 'utf8': 1, 'phone': phone, 'text': text.encode('utf-8')}),
        )
        try:
            response = requests.get(url, timeout=SMS_TIMEOUT)
            response.raise_for_status()
        except (requests.exceptions.ConnectTimeout, requests.exceptions.ReadTimeout):
            raise
        except requests.exceptions.HTTPError:
            logger.exception('Sending sms raised HTTPError')
        else:
            tree = xml.etree.ElementTree.fromstring(response.content)
            message_sent = tree.find('message-sent')
            if message_sent is not None:
                sms_id = message_sent.attrib['id']
                logger.info('Successfully sent sms to %s with id %s for answer_id %s', phone, sms_id, answer_id)
                increase_field(answer_id, 'sms_success')
            else:
                error = tree.find('error')
                if error is not None:
                    error_msg = error.text
                    logger.error('SMS to %s was not sent with error `%s`', phone, error_msg)
                else:
                    logger.error('Unknown response format from %s: %s', settings.SMS_PASSPORT_HOST, response.content,)


@celery.shared_task(bind=True)
def create_emergency_notifications_for_staff_page(self, query, answer_id, test_run, min_id, max_id):
    testing_logins = get_testing_logins(test_run)
    lookup = {
        '_query': '({}) and id > {} and id <= {}'.format(query, min_id, max_id),
        '_fields': 'location,uid,id,work_email,login,phones',
        '_limit': max_id - min_id
    }
    try:
        page = staff.persons.getiter(lookup).first_page
        contexts_dont_send = {}
        for person in page:
            uid = person['uid']
            login = person['login']
            main_phone = staff.find_main_phone(person.get('phones'))
            dont_send = False
            if test_run and login not in testing_logins:
                dont_send = True

            context = {'office': person['location']['office']['name']['ru']}
            if person['location']['office']['id'] == AURORA_OFFICE_ID:
                floor_name = person['location'].get('table', {}).get('floor', {}).get('name', {}).get('en')
                if floor_name:
                    context['office'] += ', Корпус %s' % (AURORA_SECTION_NAME_REGEXP.findall(floor_name)[0])

            if 'work_email' in person:
                email = person['work_email']
                mail_notify.delay(email, answer_id, context, dont_send)
                increase_field(answer_id, 'mail_attempts')
                logger.info('Trying to send email to %s for answer_id %s', email, answer_id)

            if main_phone:
                sms_notify.delay(main_phone, answer_id, context, dont_send)
                increase_field(answer_id, 'sms_attempts')
                logger.info('Trying to send sms to %s for answer_id %s', main_phone, answer_id)

            contexts_dont_send[int(uid)] = (context, dont_send)

        yamb_ids = {
            int(user.uid): user.yamb_id
            for user in models.User.objects.filter(uid__in=contexts_dont_send.keys()).exclude(yamb_id='')
        }

        telegram_usernames = models.TelegramUsername.objects.filter(
            user_id__in=contexts_dont_send.keys(),
            telegram_id__isnull=False
        )
        telegram_ids = collections.defaultdict(list)
        for telegram_username in telegram_usernames:
            telegram_ids[int(telegram_username.user_id)].append(telegram_username.telegram_id)

        for uid, (context, dont_send) in contexts_dont_send.items():
            for telegram_id in telegram_ids[uid]:
                telegram_notify.delay(telegram_id, answer_id, context, dont_send)
                increase_field(answer_id, 'telegram_attempts')
                logger.info('Trying to send telegram_message to %s for answer_id %s', telegram_id, answer_id)

            if uid in yamb_ids:
                q_notify.delay(yamb_ids[uid], answer_id, context, dont_send)
                increase_field(answer_id, 'yamb_attempts')
                logger.info('Trying to send q_message to %s for answer_id %s', yamb_ids[uid], answer_id)

    except (requests.RequestException, ids_exceptions.BackendError) as exc:
        logger.error('Create emergency by page failed, retrying...')
        raise self.retry(
            exc=exc,
            countdown=5 * self.request.retries,
            max_retries=None
        )
    except Exception:
        message = 'create_emergency_notifications_for_staff_page %d - %d raised exception' % (
            min_id, max_id
        )
        logger.exception(message)
        if yenv.type != 'development':
            mail.send_mail(
                settings.MAIL_FROM,
                settings.UHURA_MAIL,
                'EmergencyNotificationError: ' + message,
                template=''.join(traceback.format_exception(*sys.exc_info()))
            )


@celery.shared_task(bind=True)
def create_emergency_notifications(self, countries, offices, cities, floors, answer_id, test_run):
    query_array = []
    q = django_models.Q()
    section_floors_mapping = AURORA_SECTION_FLOORS_MAPPING.copy()
    for country in countries:
        query_array.append(COUNTRY_QUERY.format(country))
        q |= django_models.Q(country_code=country)
    for city in cities:
        query_array.append(CITY_QUERY.format(city))
        q |= django_models.Q(city_code=city)
    for office in offices:
        query_array.append(OFFICE_QUERY.format(office))
        q |= django_models.Q(office_code=office)
    for floor in floors:
        query_array.append(FLOOR_QUERY.format(floor))
        for section, floors in section_floors_mapping.iteritems():
            if floor in floors:
                recipients = [['mosoffice-announces@yandex-team.ru'], ['fireman@yandex-team.ru']]
                if test_run:
                    recipients = [[settings.UHURA_MAIL]]
                for to in recipients:
                    mail_notify.delay(
                        to,
                        answer_id,
                        {'office': 'Москва, БЦ Аврора, Корпус %s' % section},
                        False,
                        use_bcc=True
                    )
                    increase_field(answer_id, 'mail_attempts')
                    logger.info('Trying to send email to %s for answer_id %s', to, answer_id)

                section_floors_mapping.pop(section)
                break

    try:
        if q:
            for office in models.Office.objects.filter(q):
                recipients = [office.emergency_emails, ['fireman@yandex-team.ru']]
                if test_run:
                    recipients = [[settings.UHURA_MAIL]]
                for to in recipients:
                    mail_notify.delay(
                        to,
                        answer_id,
                        {'office': office.name},
                        False,
                        use_bcc=True
                    )
                    increase_field(answer_id, 'mail_attempts')
                    logger.info('Trying to send email to %s for answer_id %s', to, answer_id)

        query = '({}) and official.is_dismissed == False and official.is_robot == False'.format(
            ' or '.join(query_array)
        )
        logger.info('Notifying users with query "%s"', query)
        lookup = {
            '_query': query,
            '_fields': 'id',
            '_limit': 1,
            '_sort': '-id'
        }
        max_id = staff.persons.get_one(lookup=lookup)['id']
        for i in xrange(0, max_id, STAFF_ID_RANGE):
            j = i + STAFF_ID_RANGE
            create_emergency_notifications_for_staff_page.delay(query, answer_id, test_run, i, j)

    except (requests.RequestException, ids_exceptions.BackendError) as exc:
        logger.exception('Emergency notification creation failed, retrying')
        raise self.retry(
            exc=exc,
            countdown=5 * self.request.retries,
            max_retries=None
        )
    except DatabaseError as exc:
        logger.exception('DatabaseError during creating emergency notifications')
        raise self.retry(
            exc=exc,
            countdown=0.2 * self.request.retries,
            max_retries=100,
        )
    except Exception:
        logger.exception('create_emergency_notifications raised exception')
        if yenv.type != 'development':
            mail.send_mail(
                settings.MAIL_FROM,
                settings.UHURA_MAIL,
                'EmergencyNotificationError: create_emergency_notifications raised exception',
                template=''.join(traceback.format_exception(*sys.exc_info()))
            )
