# coding: utf-8

from datetime import datetime
import logging

from django.db.models import Q
import celery

from uhura import models
from uhura.external import staff
from uhura.lib import lock
from uhura.tasks.notifications import calendar


logger = logging.getLogger(__name__)
MAX_RETRIES = 10
TASKS_MAPPING = {
    'meetings_digest': calendar.meetings_digest
}


@celery.shared_task(time_limit=30)
@lock.locked('check_notifications')
def check_notifications():
    logger.info('Started checking notifications')
    now = datetime.now()
    for notification in models.PeriodicNotification.objects.all():
        if notification.last_success_date < now.date() and notification.time < now.time():
            TASKS_MAPPING[notification.subscription.task_name].delay(notification.id, **notification.params)
            logger.info(
                 'Started notification %s with params %r', notification.subscription.task_name, notification.params
            )
    logger.info('Finished checking notifications')


@celery.shared_task
@lock.locked('clean_notification_table')
def clean_notification_table():
    logger.info('Started notification table cleaning')
    uids = models.PeriodicNotification.objects.distinct('user').only('user').values_list('user__uid', flat=True)
    q = Q()
    for uid in uids:
        staff_data = staff.get_person_data_by_uid(uid)
        if staff_data['official']['is_dismissed']:
            q |= Q(user__uid=uid)
    if q:
        models.PeriodicNotification.objects.filter(q).delete()
    logger.info('Finished notification table cleaning')
