# coding: utf-8

from __future__ import unicode_literals

from datetime import datetime, date, timedelta
import logging

import celery

from vins_core.dm.request import ReqInfo, AppInfo, Utterance

from uhura import models
from uhura.app import app_connector
from uhura.external import calendar
from uhura.lib import callbacks
from uhura.lib.vins.response import UhuraVinsResponse


logger = logging.getLogger(__name__)


MAX_RETRIES = 10
EMPTY_UTTERANCE = Utterance('')
APP_INFO = AppInfo(app_id='uhura')


@celery.shared_task(bind=True)
def meetings_digest(self, notification_id, chat_id, uid, _type):
    models.PeriodicNotification.objects.filter(id=notification_id).update(last_success_date=date.today())
    prev_response = None
    req_info = ReqInfo(uuid=chat_id, utterance=EMPTY_UTTERANCE, client_time=datetime.now(), app_info=APP_INFO)
    session = app_connector.vins_app._load_session(req_info)
    form = session.form
    if form is not None:
        response_dict = models.Session.objects.get(
            uuid=chat_id
        ).session['objects']['dialog_history']['value']['turns'][-1]['response']
        if not response_dict.get('vins_response_overrided'):
            response_dict['messages'] = [response_dict['voice_text']]
        prev_response = UhuraVinsResponse.from_dict(response_dict)
        prev_response.messages = ['Прости, пришлось прервать наш разговор сообщением из подписки. '
                                  'Теперь мы можем продолжить там же, где остановились!'] + prev_response.messages
        if prev_response.inline_keyboards is not None:
            prev_response.inline_keyboards = [None] + prev_response.inline_keyboards
        prev_response.vins_response_overrided = True
    if _type == 'today':
        from_date = datetime.now()
    else:
        from_date = datetime.now() + timedelta(days=1)
    if from_date.date() not in calendar.get_holidays():
        user = models.PeriodicNotification.objects.get(id=notification_id).user
        credentials = calendar.CalendarRequestCredentials({'uid': user.uid, 'login': user.username})
        request = calendar.MeetingsRequest(uid, from_date, None, credentials)
        events = callbacks.get_meetings_by_date(request)
        if events is None:
            raise self.retry(
                countdown=2 ** self.request.retries,
                max_retries=MAX_RETRIES
            )
        calendar.cast_timestamps_to_readable_format(events)
        context = {
            'events': events,
            'from_date': from_date,
            'digest_type': _type
        }
        response = app_connector.vins_app.generate_notification('get_meetings_by_date', 'calendar', 'meetings', context)
        response.messages = response.messages[0].split('\n\n')
        app_connector.handle_response(chat_id, response)
        if prev_response:
            app_connector.handle_response(chat_id, prev_response)
