# coding: utf-8

from __future__ import unicode_literals

from datetime import timedelta
import logging

import celery

from django.conf import settings
from django.utils import timezone

from vins_core.dm import session as vins_session
from vins_core.dm import request as vins_request

from uhura import models
from uhura.app import app_connector
from uhura.lib import lock
from uhura.lib.vins.response import UhuraVinsResponse


logger = logging.getLogger(__name__)
CLOSE_SESSION_RESPONSE = UhuraVinsResponse()
CLOSE_SESSION_RESPONSE.reply('Ты слишком долго не отвечал на вопрос, поэтому я завершаю сессию')
MAX_RETRIES = 5
NOT_FINISHED_INTENTS = ['uhura.quest.{}'.format(x) for x in settings.QUEST_TASKS]


def _create_dummy_req_info(uuid):
    return vins_request.create_request(uuid, utterance='')


@celery.shared_task(autoretry_for=(Exception,), retry_kwargs={'max_retries': MAX_RETRIES, 'countdown': 2})
@lock.locked('unlock_all_cars')
def unlock_all_cars():
    logger.info('Started unlocking all cars')
    models.User.objects.update(blocked_cars_plates=list())
    logger.info('Finished unlocking all cars')


@celery.shared_task
@lock.locked('close_old_sessions')
def close_old_sessions():
    logger.info('Started closing sessions')
    sessions_to_close = models.Session.objects.filter(
        updated_at__lte=timezone.now().replace(second=0, microsecond=0) - timedelta(seconds=settings.SESSION_TIMEOUT),
        session__objects__form__value__form__in=NOT_FINISHED_INTENTS
    )
    for session_obj in sessions_to_close:
        try:
            models.User.objects.get(telegram_usernames__telegram_id=session_obj.uuid)
        except models.User.DoesNotExist:
            pass
        else:
            session = vins_session.Session.from_dict(session_obj.session)
            uuid = session_obj.uuid
            req_info = _create_dummy_req_info(uuid)
            app_connector.handle_response(session_obj.uuid, CLOSE_SESSION_RESPONSE)
            app_connector.vins_app.change_intent_to_empty(
                session,
                req_info,
                CLOSE_SESSION_RESPONSE
            )
            app_connector.vins_app.save_session(session, req_info)
    logger.info('Finished closing sessions')
