# coding: utf-8
from __future__ import unicode_literals

import datetime
from enum import IntEnum
import pytz

from vins_core.nlg.filters import inflect


class Weekdays(IntEnum):
    MONDAY = 0
    TUESDAY = 1
    WEDNESDAY = 2
    THURSDAY = 3
    FRIDAY = 4
    SATURDAY = 5
    SUNDAY = 6

    @staticmethod
    def range(start, stop):
        range = []
        x = start
        while x != stop + 1:
            range.append(x)
            x += 1
            if x > Weekdays.SUNDAY:
                x = Weekdays.MONDAY
        return range


def dict_to_datetime(dct):
    return datetime.datetime(
        year=dct.get('years', 0),
        month=dct.get('months', 0),
        day=dct.get('days', 0),
        hour=dct.get('hours', 0),
        minute=dct.get('minutes', 0),
        second=dct.get('seconds', 0),
    )


def dict_to_date(dct):
    return datetime.date(
        year=dct.get('years', 0),
        month=dct.get('months', 0),
        day=dct.get('days', 0),
    )


def dict_to_time(dct):
    return datetime.time(
        hour=dct.get('hours', 0),
        minute=dct.get('minutes', 0),
        second=dct.get('seconds', 0)
    )


def parse_date_dict(dct, period_end=False):
    if 'weekday' in dct:
        return next_weekday(datetime.date.today(), dct['weekday'] - 1)

    if 'days_relative' in dct:
        return datetime.date.today() + datetime.timedelta(days=dct['days'])

    if 'weeks_relative' in dct:
        if period_end:
            return prev_weekday(datetime.date.today(), Weekdays.MONDAY)\
                   + datetime.timedelta(weeks=dct['weeks'], days=-1)
        else:
            return prev_weekday(datetime.date.today(), Weekdays.MONDAY) + datetime.timedelta(weeks=dct['weeks'])

    if 'years_relative' in dct:
        dct.pop('years_relative')
        dct.pop('years')
        return parse_date_dict(dct)

    if 'weekend' in dct:
        if period_end:
            return next_weekday(datetime.date.today(), Weekdays.SUNDAY)
        else:
            return next_weekday(datetime.date.today(), Weekdays.SATURDAY)

    if 'years' not in dct:
        return dict_to_date(add_year(dct))

    return dict_to_date(dct)


def datetime_range_to_datetimes(dct):
    from_date, to_date = parse_date_dict(dct['start']), parse_date_dict(dct['end'], period_end=True)
    if ('weekday' in dct['start'] or 'weekday' in dct['end']) and from_date > to_date:
        to_date += datetime.timedelta(weeks=1)
    return from_date, to_date


def next_weekday(date, weekday):
    days_ahead = weekday - date.weekday()
    if days_ahead < 0:
        days_ahead += 7
    return date + datetime.timedelta(days_ahead)


def prev_weekday(date, weekday):
    days_behind = date.weekday() - weekday
    if days_behind < 0:
        days_behind += 7
    return date - datetime.timedelta(days_behind)


def last_day_of_month(month_number):
    date = datetime.date.today().replace(month=month_number)
    next_month = date.replace(day=28) + datetime.timedelta(days=4)
    return next_month - datetime.timedelta(days=next_month.day)


def add_year(dct):
    dct['years'] = datetime.datetime.now().year
    date = dict_to_date(dct)
    if date < datetime.date.today() and datetime.date.today().month - date.month > 3:
        dct['years'] += 1
    return dct


def process_date_range(date_range, value_type):
    if value_type == 'datetime_range':
        from_date, to_date = datetime_range_to_datetimes(date_range)

    else:
        if len(date_range) == 1 and 'months' in date_range:
            month_number = date_range['months']
            from_date = datetime.date.today().replace(month=month_number, day=1)
            to_date = last_day_of_month(month_number)
            if to_date < datetime.date.today():
                from_date = from_date.replace(year=from_date.year + 1)
                to_date = to_date.replace(year=to_date.year + 1)

        elif len(date_range) == 1 and 'years' in date_range:
            year = date_range['years']
            from_date = datetime.date(year=year, month=1, day=1)
            to_date = datetime.date(year=year, month=12, day=31)

        else:
            from_date = parse_date_dict(date_range)
            to_date = None

    return from_date, to_date


def convert_datetime(d, from_tz, to_tz):
    from_tz = pytz.timezone(from_tz)
    to_tz = pytz.timezone(to_tz)
    return from_tz.localize(d).astimezone(to_tz)


def convert_utc_to_usertz(date, user_tz):
    # type(str, str) -> str
    return datetime_to_str(convert_datetime(str_to_datetime(date), 'UTC', user_tz))


def str_to_datetime(datetime_str):
    return datetime.datetime.strptime(datetime_str, '%Y-%m-%dT%H:%M:%S')


def datetime_to_str(date):
    return date.strftime('%Y-%m-%dT%H:%M:%S')


def datetime_to_readable_format(date):
    now = datetime.datetime.now()
    res_dict = {'day': ''}

    if date.date() == now.date():
        res_dict['day'] = 'Сегодня'
    elif date.date() == now.date() + datetime.timedelta(days=1):
        res_dict['day'] = 'Завтра'
    elif date.date == now.date() - datetime.timedelta(days=1):
        res_dict['day'] = 'Вчера'

    date_string = '{} {}'.format(date.day, inflect(date.strftime('%B').decode('utf-8'), 'gen'))
    if res_dict['day']:
        res_dict['day'] += ' ({})'.format(date_string)
    else:
        res_dict['day'] = date_string
        if date.year != now.year:
            res_dict['day'] += ' {}'.format(date.year)

    res_dict['time'] = date.strftime('%H:%M')
    return res_dict


def datetime_str_to_readable_format(datetime_str):
    return datetime_to_readable_format(str_to_datetime(datetime_str))
