# coding: utf-8

from __future__ import unicode_literals

import json
import logging
import os

from django import http
from django.conf import settings
from django import forms
from django.shortcuts import render
from django.utils import timezone
from django.views.generic import base

from blackbox import Blackbox
from telegram import (
    Update as TelegramUpdate,
    Bot,
)
from ticket_parser2.api.v1 import BlackboxClientId
import waffle
import yenv

from uhura import app
from uhura import models
from uhura.external import forms as yandex_forms
from uhura.lib import cache_storage
from uhura.lib.yamb import Update as YambUpdate
from uhura.models import TelegramUsername
from uhura.models import SyncTime
from uhura.tasks import emergency
from uhura.utils import hasher
from uhura.utils.tvm import get_tvm_service_headers

logger = logging.getLogger(__name__)
UPDATE_CACHE_TIME = 30


class WebhookView(base.View):
    bot_token = None
    update_class = None

    def post(self, request, token_hash):
        if hasher.sha256(self.bot_token) != token_hash:
            return http.HttpResponseForbidden()

        try:
            json_body = json.loads(request.body.decode('utf-8'))
        except ValueError:
            return http.HttpResponseBadRequest('Bad json object')

        status_code = 200
        if waffle.switch_is_active('process_messages'):
            update_id = None
            try:
                update = self.update_class.de_json(json_body, app.app_connector.bot)
                update_id = update.update_id
                if cache_storage.get(update_id) != cache_storage.EXPIRED_VALUE:
                    return http.HttpResponse()
                cache_storage.set(update_id, update_id, UPDATE_CACHE_TIME)
                app.app_connector.bot_handler(app.app_connector.bot, update)
            except app.TelegramRetry:
                status_code = 500
            except Exception:
                logger.exception('Exception while processing update:')
            if update_id:
                cache_storage.set(update_id, cache_storage.EXPIRED_VALUE, UPDATE_CACHE_TIME)
        return http.HttpResponse(status=status_code)

    @classmethod
    def get_webhook_view(cls):
        if os.environ.get('YAMB_BOT'):
            return YambWebhookView

        return TelegramWebhookView


class TelegramWebhookView(WebhookView):
    bot_token = settings.TELEGRAM_TOKEN
    update_class = TelegramUpdate


class YambWebhookView(WebhookView):
    bot_token = settings.YAMB_TOKEN
    update_class = YambUpdate


class PingView(base.View):
    def get(self, request):
        return http.HttpResponse('pong')


class OAuthTokenForm(forms.Form):
    access_token = forms.CharField(min_length=39, max_length=39)


class PassportTokenCallbackView(base.View):
    def get(self, request):
        return render(request, 'oauth_callback.html')

    def _get_telegram_id(self, uid):
        telegram_ids = list(
            models.TelegramUsername.objects
            .filter(user_id=uid, telegram_id__isnull=False)
            .values_list('telegram_id', flat=True)
        )
        if not telegram_ids:
            logger.warning('Not found telegram_id for uid %s', uid)
            return
        if len(telegram_ids) > 1:
            logger.warning(
                'Found several telegram ids %s for uid %s. Going to use the first one.',
                telegram_ids,
                uid,
            )
        return telegram_ids[0]

    def post(self, request):
        form = OAuthTokenForm(request.POST)
        if not form.is_valid():
            return http.HttpResponseBadRequest()

        token = form.cleaned_data['access_token']
        bb_answer = Blackbox().oauth(
            get_tvm_service_headers(BlackboxClientId.ProdYateam.value),
            userip='::1',
            oauth_token=token,
        )

        if bb_answer.get('status') != 'VALID':
            return http.HttpResponseBadRequest()

        uid = bb_answer['uid']
        models.UserToken.objects.update_or_create(
            user_id=uid,
            defaults={'token': token, 'user_id': uid},
        )

        telegram_id = self._get_telegram_id(uid)
        if not telegram_id:
            return http.HttpResponseBadRequest()
        form = app.app_connector.vins_app.new_form('empty_intent', 'utils')
        text = app.app_connector.vins_app.render_phrase('token_added', form=form).text
        app.app_connector.bot.send_message(chat_id=telegram_id, text=text)

        redirect_url = 'tg://resolve?domain={}'.format(settings.TELEGRAM_LOGIN)
        return http.HttpResponse(
            json.dumps({'redirect_url': redirect_url}),
            status=201,
            content_type="application/json",
        )


class PendingUpdateCountMonitoringView(base.View):
    def get(self, request):
        count = app.app_connector.bot.get_webhook_info().to_dict()['pending_update_count']
        response = http.HttpResponse(content='%d pending updates' % count)
        if count > settings.MAX_PENDING_REQUESTS:
            response.status_code = 500
        return response


class NotPushedMonitoringView(base.View):
    def get(self, request):
        not_pushed = TelegramUsername.objects.filter(
            pushed_to_tasha=False,
            updated_at__lt=timezone.now() - timezone.timedelta(minutes=10)
        )

        return http.JsonResponse(
            {'not_pushed_telegrams': list(not_pushed.values_list('telegram_id', flat=True))},
            status=500 if not_pushed.count() else 200,
        )


def _parse_form(form_body):
    form = yandex_forms.Form(settings.EMERGENCY_FORM)
    submitted_form_dict = {}
    for line in form_body.split('\r'):
        try:
            field_json = json.loads(line)
            if field_json.get('value'):
                label = field_json['question']['label']['ru']
                for field_id, field in form:
                    if field.get_label() == label:
                        if isinstance(field, yandex_forms.SelectInput):
                            values = field_json['value'].split(', ')
                            processed_values = []
                            for value in values:
                                for option in field.get_options_dict():
                                    if value == option['text']:
                                        processed_values.append(option['slug'])
                                        break
                            submitted_form_dict[field_id] = processed_values
                        else:
                            submitted_form_dict[field_id] = field_json['value']
        except ValueError:
            continue
    return submitted_form_dict


class EmergencySituationView(base.View):
    def post(self, request):
        author = request.META.get('HTTP_X_AUTHOR')
        if author is None:
            logger.error('No author passed to /emergency/')
            return http.HttpResponseBadRequest('"login" should be not none')

        try:
            models.EmergencyNotifier.objects.get(staff_login=author)
        except models.EmergencyNotifier.DoesNotExist:
            return http.HttpResponseForbidden('You can\'t send emergency notifications')

        country_codes = models.Office.objects.distinct('country_code').values_list('country_code', flat=True)
        city_codes = models.Office.objects.distinct('city_code').values_list('city_code', flat=True)
        office_codes = models.Office.objects.values_list('office_code', flat=True)

        country_city_mapping = {
            country_code: models.Office.objects.filter(
                country_code=country_code
            ).distinct('city_code').values_list('city_code', flat=True) for country_code in country_codes
        }
        city_office_mapping = {
            city_code: models.Office.objects.filter(
                city_code=city_code
            ).values_list('office_code', flat=True) for city_code in city_codes
        }
        office_floor_mapping = {
            office_code: models.Office.objects.get(office_code=office_code).floor_codes for office_code in office_codes
        }

        answer_id = request.META['HTTP_X_FORM_ANSWER_ID']
        parsed_form = _parse_form(request.body.decode('utf-8'))

        countries = filter(None, parsed_form.get('country', '').split(', '))
        cities = []
        offices = []
        floors = []
        long_template = None
        short_template = None
        title = None
        for k, v in parsed_form.iteritems():
            if k.startswith('title__'):
                title = v
            elif k.startswith('content_short__'):
                short_template = v
            elif k.startswith('content__'):
                long_template = v
            elif k.startswith('city__'):
                cities.extend(v)
            elif k.startswith('office__'):
                offices.extend(v)
            elif k.startswith('country__'):
                countries.extend(v)
            elif k.startswith('floors__'):
                for x in v:
                    floors.extend(x.split('_'))

        if not all((long_template, short_template, title)):
            return http.HttpResponseBadRequest('Long_template, short_template and title should be provided')

        for country in list(countries):
            for country_city in country_city_mapping[country]:
                if country_city in cities:
                    countries.remove(country)
                    break
        for city in list(cities):
            for city_office in city_office_mapping[city]:
                if city_office in offices:
                    cities.remove(city)
                    break
        for office in list(offices):
            for office_floor in office_floor_mapping[office]:
                if office_floor in floors:
                    offices.remove(office)
                    break
        if not any((countries, cities, offices, floors)):
            countries = country_city_mapping.keys()

        test_run = (waffle.switch_is_active('emergency_test_run') or yenv.type != 'production')
        notification_data, created = models.EmergencyNotification.objects.get_or_create(
            answer_id=answer_id,
            defaults={
                'author': author,
                'long_template': long_template,
                'short_template': short_template,
                'subject': title
            }
        )
        if created:
            logger.info('Received emergency notification %s', answer_id)

            emergency.create_emergency_notifications.delay(countries, offices, cities, floors, answer_id, test_run)
        return http.HttpResponse()


class StaffSyncMonitoring(base.View):
    SECONDS_IN_HOUR = 60 * 60

    def get(self, request):
        last_sync_hours = timezone.timedelta(hours=0)
        try:
            sync_time = SyncTime.objects.get(name='import_staff')
        except SyncTime.DoesNotExist:
            status = 500
        else:
            last_sync_hours = timezone.now() - sync_time.last_success_start
            status = 200 if last_sync_hours < timezone.timedelta(hours=1) else 500
        return http.HttpResponse (
            content='Last sync was finished %.1f hours ago' %
                    (last_sync_hours.total_seconds() / self.SECONDS_IN_HOUR),
            status=status
        )
