package ru.yandex.galois

import kotlinx.coroutines.runBlocking
import mu.KotlinLogging
import org.springframework.boot.CommandLineRunner
import org.springframework.stereotype.Component
import ru.yandex.galois.clients.abc.AbcClient
import ru.yandex.galois.clients.idm.IdmClient
import ru.yandex.galois.clients.staff.StaffClient
import ru.yandex.galois.clients.yt.YtWriter

private val logger = KotlinLogging.logger {}

@Component
class Runner(private val abcClient: AbcClient,
    private val idmClient: IdmClient,
    private val staffClient: StaffClient,
    private val ytWriter: YtWriter): CommandLineRunner {

    override fun run(vararg args: String?) {
        logger.info { "Exporting IDM groups..." }
        runBlocking {
            val idmGroupsFlow = idmClient.getAllGroups(args[0]!!, "service", 1000)
            ytWriter.writeIdmGroups(args[1]!!, args[2]!!, idmGroupsFlow)
        }
        logger.info { "IDM groups exported..." }
        logger.info { "Exporting ABC services..." }
        runBlocking {
            val abcServicesFlow = abcClient.getAllServices(args[0]!!, 1000, "id", true)
            ytWriter.writeAbcServices(args[1]!!, args[2]!!, abcServicesFlow)
        }
        logger.info { "ABC services exported..." }
        logger.info { "Exporting ABC roles..." }
        runBlocking {
            val abcRolesFlow = abcClient.getAllRoles(args[0]!!, 1000, "id")
            ytWriter.writeAbcRoles(args[1]!!, args[2]!!, abcRolesFlow)
        }
        logger.info { "ABC roles exported..." }
        logger.info { "Exporting ABC service members..." }
        runBlocking {
            val abcServiceMembersFlow = abcClient.getAllServiceMembers(args[0]!!, 1000, "id", true)
            ytWriter.writeAbcServiceMembers(args[1]!!, args[2]!!, abcServiceMembersFlow)
        }
        logger.info { "ABC service members exported..." }
        logger.info { "Exporting Staff groups..." }
        runBlocking {
            val staffServiceGroupsFlow = staffClient.getAllGroups(args[0]!!, "service", 1000, "id", true)
            val staffServiceRoleGroupsFlow = staffClient.getAllGroups(args[0]!!, "servicerole", 1000, "id", true)
            ytWriter.writeStaffGroups(args[1]!!, args[2]!!, listOf(staffServiceGroupsFlow, staffServiceRoleGroupsFlow))
        }
        logger.info { "Staff groups exported..." }
        logger.info { "Exporting Staff group memberships..." }
        runBlocking {
            val staffServiceGroupMembershipsFlow = staffClient.getAllGroupMemberships(args[0]!!, "service",
                1000, "id")
            val staffServiceRoleGroupMembershipsFlow = staffClient.getAllGroupMemberships(args[0]!!, "servicerole",
                1000, "id")
            ytWriter.writeStaffGroupMemberships(args[1]!!, args[2]!!,
                listOf(staffServiceGroupMembershipsFlow, staffServiceRoleGroupMembershipsFlow))
        }
        logger.info { "Staff group memberships exported..." }
    }

}
