package ru.yandex.galois.clients.staff

import com.fasterxml.jackson.databind.ObjectMapper
import io.github.resilience4j.kotlin.retry.executeSuspendFunction
import io.github.resilience4j.retry.Retry
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.beans.factory.annotation.Value
import org.springframework.http.HttpHeaders
import org.springframework.stereotype.Component
import org.springframework.web.reactive.function.client.WebClient
import org.springframework.web.reactive.function.client.awaitEntity
import org.springframework.web.reactive.function.client.awaitExchange
import org.springframework.web.reactive.function.client.createExceptionAndAwait
import org.springframework.web.util.UriComponentsBuilder
import ru.yandex.galois.clients.staff.model.StaffGroup
import ru.yandex.galois.clients.staff.model.StaffGroupMembership
import ru.yandex.galois.clients.staff.model.StaffResponse
import ru.yandex.galois.clients.utils.ClientFactory
import java.util.concurrent.atomic.AtomicReference

@Component
class StaffClient(@Value("\${staff.client.connectTimeoutMs}") connectTimeoutMillis: Int,
    @Value("\${staff.client.readTimeoutMs}") readTimeoutMillis: Long,
    @Value("\${staff.client.writeTimeoutMs}") writeTimeoutMillis: Long,
    @Value("\${staff.client.sslHandshakeTimeoutMs}") sslHandshakeTimeoutMillis: Long,
    @Value("\${staff.client.sslCloseNotifyReadTimeoutMs}") sslCloseNotifyReadTimeoutMillis: Long,
    @Value("\${staff.client.sslCloseFlushTimeoutMs}") sslCloseFlushTimeoutMillis: Long,
    @Value("\${staff.client.responseTimeoutMs}") responseTimeoutMillis: Long,
    @Value("\${staff.client.baseUrl}") private val baseUrl: String,
    @Value("\${staff.client.maxResponseBytes}") maxResponseBytes: Long,
    @Value("\${http.client.userAgent}") private val userAgent: String,
    @Qualifier("httpClientObjectMapper") objectMapper: ObjectMapper,
    @Qualifier("httpRetry") private val retry: Retry
) {

    private val client: WebClient

    init {
        client = ClientFactory.prepareClient(connectTimeoutMillis, readTimeoutMillis, writeTimeoutMillis,
            sslHandshakeTimeoutMillis, sslCloseNotifyReadTimeoutMillis, sslCloseFlushTimeoutMillis,
            responseTimeoutMillis, objectMapper, maxResponseBytes)
    }

    suspend fun getGroupMembership(oauthToken: String, groupType: String? = null, limit: Long? = null,
        page: Long? = null, sort: String? = null): StaffResponse<StaffGroupMembership> {
        val fields = listOf("group.id", "group.is_deleted", "group.url", "group.type", "group.role_scope",
            "group.service.id", "group.parent.id", "group.parent.is_deleted", "group.parent.url", "group.parent.type",
            "group.parent.role_scope", "group.parent.service.id", "_meta.modified_at", "joined_at",
            "person.is_deleted", "person.official.is_dismissed", "person.login", "person.id", "id")
            .joinToString(separator = ",")
        val uriBuilder = UriComponentsBuilder
            .fromHttpUrl(baseUrl)
            .path("/groupmembership/")
            .queryParam("_fields", fields)
        limit?.let { uriBuilder.queryParam("_limit", it) }
        page?.let { uriBuilder.queryParam("_page", it) }
        sort?.let { uriBuilder.queryParam("_sort", it) }
        groupType?.let { uriBuilder.queryParam("group.type", it) }
        val uri = uriBuilder.build().toUri()
        return retry.executeSuspendFunction {
            client.get()
                .uri(uri)
                .header("Authorization", "OAuth $oauthToken")
                .header(HttpHeaders.USER_AGENT, userAgent)
                .awaitExchange { response ->
                    if (response.statusCode().is2xxSuccessful) {
                        response.awaitEntity<StaffResponse<StaffGroupMembership>>().body
                    } else {
                        throw response.createExceptionAndAwait()
                    }
                }
        }
    }

    suspend fun getGroups(oauthToken: String, type: String? = null, limit: Long? = null, page: Long? = null,
        sort: String? = null, withDeleted: Boolean = false): StaffResponse<StaffGroup> {
        val fields = listOf("id", "is_deleted", "url", "type", "role_scope", "service.id", "parent.id",
            "parent.is_deleted", "parent.url", "parent.type", "parent.role_scope", "parent.service.id")
            .joinToString(separator = ",")
        val uriBuilder = UriComponentsBuilder
            .fromHttpUrl(baseUrl)
            .path("/groups/")
            .queryParam("_fields", fields)
        limit?.let { uriBuilder.queryParam("_limit", it) }
        page?.let { uriBuilder.queryParam("_page", it) }
        sort?.let { uriBuilder.queryParam("_sort", it) }
        type?.let { uriBuilder.queryParam("type", it) }
        if (withDeleted) {
            uriBuilder.queryParam("_query", "is_deleted in [false, true]")
        }
        val uri = uriBuilder.build().toUri()
        return retry.executeSuspendFunction {
            client.get()
                .uri(uri)
                .header("Authorization", "OAuth $oauthToken")
                .header(HttpHeaders.USER_AGENT, userAgent)
                .awaitExchange { response ->
                    if (response.statusCode().is2xxSuccessful) {
                        response.awaitEntity<StaffResponse<StaffGroup>>().body
                    } else {
                        throw response.createExceptionAndAwait()
                    }
                }
        }
    }

    suspend fun getAllGroups(oauthToken: String, type: String? = null, pageSize: Long? = 1000,
        sort: String? = null, withDeleted: Boolean = false): Flow<List<StaffGroup>> = flow {
        val firstPage = getGroups(oauthToken = oauthToken, type = type, limit = pageSize, sort = sort,
            withDeleted = withDeleted)
        if (firstPage.result.isNotEmpty()) {
            emit(firstPage.result)
        }
        val lastNext = AtomicReference(firstPage.links.next)
        while (true) {
            val nextPageNumber = nextPage(lastNext.get()) ?: break
            val nextPage = getGroups(oauthToken = oauthToken, type = type, limit = pageSize, page = nextPageNumber,
                sort = sort, withDeleted = withDeleted)
            if (nextPage.result.isNotEmpty()) {
                emit(nextPage.result)
            }
            lastNext.set(nextPage.links.next)
        }
    }

    suspend fun getAllGroupMemberships(oauthToken: String, groupType: String? = null, pageSize: Long? = 1000,
        sort: String? = null): Flow<List<StaffGroupMembership>> = flow {
        val firstPage = getGroupMembership(oauthToken = oauthToken, groupType = groupType, limit = pageSize,
            sort = sort)
        if (firstPage.result.isNotEmpty()) {
            emit(firstPage.result)
        }
        val lastNext = AtomicReference(firstPage.links.next)
        while (true) {
            val nextPageNumber = nextPage(lastNext.get()) ?: break
            val nextPage = getGroupMembership(oauthToken = oauthToken, groupType = groupType, limit = pageSize,
                page = nextPageNumber, sort = sort)
            if (nextPage.result.isNotEmpty()) {
                emit(nextPage.result)
            }
            lastNext.set(nextPage.links.next)
        }
    }

    private fun nextPage(nextUrl: String?): Long? {
        return UriComponentsBuilder.fromHttpUrl(nextUrl ?: return null).build()
            .queryParams.getFirst("_page")?.toLongOrNull()
    }

}
