package ru.yandex.galois.clients.utils

import com.fasterxml.jackson.databind.ObjectMapper
import io.netty.channel.ChannelOption
import io.netty.handler.timeout.ReadTimeoutHandler
import io.netty.handler.timeout.WriteTimeoutHandler
import org.springframework.http.MediaType
import org.springframework.http.client.reactive.ReactorClientHttpConnector
import org.springframework.http.codec.ClientCodecConfigurer
import org.springframework.http.codec.json.Jackson2JsonDecoder
import org.springframework.http.codec.json.Jackson2JsonEncoder
import org.springframework.web.reactive.function.client.ExchangeFilterFunctions
import org.springframework.web.reactive.function.client.ExchangeStrategies
import org.springframework.web.reactive.function.client.WebClient
import reactor.netty.Connection
import reactor.netty.http.client.HttpClient
import reactor.netty.tcp.TcpSslContextSpec
import java.time.Duration
import java.util.concurrent.TimeUnit

object ClientFactory {

    fun prepareClient(connectTimeoutMillis: Int,
        readTimeoutMillis: Long,
        writeTimeoutMillis: Long,
        sslHandshakeTimeoutMillis: Long,
        sslCloseNotifyReadTimeoutMillis: Long,
        sslCloseFlushTimeoutMillis: Long,
        responseTimeoutMillis: Long,
        objectMapper: ObjectMapper,
        maxResponseByteCount: Long): WebClient {
        val httpClient = HttpClient.create()
            .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, connectTimeoutMillis)
            .resolver(Resolver.RESOLVER_INSTANCE)
            .responseTimeout(Duration.ofMillis(responseTimeoutMillis))
            .secure { spec -> spec.sslContext(TcpSslContextSpec.forClient())
                .handshakeTimeoutMillis(sslHandshakeTimeoutMillis)
                .closeNotifyReadTimeoutMillis(sslCloseNotifyReadTimeoutMillis)
                .closeNotifyFlushTimeoutMillis(sslCloseFlushTimeoutMillis) }
            .doOnConnected { connection: Connection ->
                connection.addHandlerLast(ReadTimeoutHandler(readTimeoutMillis, TimeUnit.MILLISECONDS))
                connection.addHandlerLast(WriteTimeoutHandler(writeTimeoutMillis, TimeUnit.MILLISECONDS))
            }
        val strategies = ExchangeStrategies
            .builder()
            .codecs { clientDefaultCodecsConfigurer: ClientCodecConfigurer ->
                clientDefaultCodecsConfigurer.defaultCodecs()
                    .jackson2JsonEncoder(Jackson2JsonEncoder(objectMapper, MediaType.APPLICATION_JSON))
                clientDefaultCodecsConfigurer.defaultCodecs()
                    .jackson2JsonDecoder(Jackson2JsonDecoder(objectMapper, MediaType.APPLICATION_JSON))
                clientDefaultCodecsConfigurer.defaultCodecs().maxInMemorySize(Math.toIntExact(maxResponseByteCount))
            }.build()
        return WebClient.builder()
            .clientConnector(ReactorClientHttpConnector(httpClient))
            .filter(ExchangeFilterFunctions.limitResponseSize(maxResponseByteCount))
            .exchangeStrategies(strategies)
            .codecs { configurer: ClientCodecConfigurer ->
                configurer
                    .defaultCodecs()
                    .maxInMemorySize(Math.toIntExact(maxResponseByteCount))
            }.build()
    }

}
