package ru.yandex.galois.clients.yt

import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.future.await
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Component
import ru.yandex.galois.clients.abc.model.AbcRole
import ru.yandex.galois.clients.abc.model.AbcService
import ru.yandex.galois.clients.abc.model.AbcServiceMember
import ru.yandex.galois.clients.idm.model.IdmGroup
import ru.yandex.galois.clients.staff.model.StaffGroup
import ru.yandex.galois.clients.staff.model.StaffGroupMembership
import ru.yandex.galois.clients.yt.model.YtAbcRole
import ru.yandex.galois.clients.yt.model.YtAbcService
import ru.yandex.galois.clients.yt.model.YtAbcServiceMember
import ru.yandex.galois.clients.yt.model.YtIdmGroup
import ru.yandex.galois.clients.yt.model.YtStaffGroup
import ru.yandex.galois.clients.yt.model.YtStaffGroupMembership
import ru.yandex.inside.yt.kosher.cypress.YPath
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.serializers.YTreeObjectSerializerFactory
import ru.yandex.misc.lang.number.UnsignedLong
import ru.yandex.type_info.TiType
import ru.yandex.yt.ytclient.`object`.MappedRowSerializer
import ru.yandex.yt.ytclient.proxy.YtClient
import ru.yandex.yt.ytclient.proxy.request.ObjectType
import ru.yandex.yt.ytclient.proxy.request.StartTransaction
import ru.yandex.yt.ytclient.proxy.request.WriteTable
import ru.yandex.yt.ytclient.rpc.RpcCredentials
import ru.yandex.yt.ytclient.rpc.RpcOptions
import ru.yandex.yt.ytclient.tables.TableSchema
import java.time.Duration
import java.time.Instant
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicBoolean

private val logger = KotlinLogging.logger {}

@Component
class YtWriter(@Value("\${yt.cluster}") private val cluster: String,
    @Value("\${yt.path}") private val path: String,
    @Value("\${yt.requestTimeoutMs}") private val requestTimeoutMillis: Long,
    @Value("\${yt.streamingReadTimeoutMinutes}") private val streamingReadTimeoutMinutes: Long,
    @Value("\${yt.streamingWriteTimeoutMinutes}") private val streamingWriteTimeoutMinutes: Long) {

    private val requestTimeout: Duration = Duration.ofMillis(requestTimeoutMillis)
    private val streamingReadTimeout: Duration = Duration.ofMinutes(streamingReadTimeoutMinutes)
    private val streamingWriteTimeout: Duration = Duration.ofMinutes(streamingWriteTimeoutMinutes)
    private val idmGroupsTablePath: String = path + "idm_groups"
    private val staffGroupsTablePath: String = path + "staff_groups"
    private val staffGroupMembershipsTablePath: String = path + "staff_group_memberships"
    private val abcServicesTablePath: String = path + "abc_services"
    private val abcRolesTablePath: String = path + "abc_roles"
    private val abcServiceMembersTablePath: String = path + "abc_service_members"
    private val idmGroupsTableSchema: TableSchema = TableSchema.builder()
        .addValue("id", TiType.int64())
        .addValue("created_at", TiType.timestamp())
        .addValue("slug", TiType.utf8())
        .addValue("state", TiType.utf8())
        .addValue("type", TiType.utf8())
        .addValue("updated_at", TiType.timestamp())
        .setUniqueKeys(false)
        .build()
    private val staffGroupsTableSchema: TableSchema = TableSchema.builder()
        .addValue("id", TiType.int64())
        .addValue("is_deleted", TiType.bool())
        .addValue("url", TiType.utf8())
        .addValue("type", TiType.utf8())
        .addValue("role_scope", TiType.optional(TiType.utf8()))
        .addValue("service_id", TiType.optional(TiType.int64()))
        .addValue("parent_id", TiType.optional(TiType.int64()))
        .addValue("parent_is_deleted", TiType.optional(TiType.bool()))
        .addValue("parent_url", TiType.optional(TiType.utf8()))
        .addValue("parent_type", TiType.optional(TiType.utf8()))
        .addValue("parent_role_scope", TiType.optional(TiType.utf8()))
        .addValue("parent_service_id", TiType.optional(TiType.int64()))
        .setUniqueKeys(false)
        .build()
    private val staffGroupMembershipsTableSchema: TableSchema = TableSchema.builder()
        .addValue("id", TiType.int64())
        .addValue("modified_at", TiType.timestamp())
        .addValue("joined_at", TiType.timestamp())
        .addValue("group_id", TiType.int64())
        .addValue("group_is_deleted", TiType.bool())
        .addValue("group_url", TiType.utf8())
        .addValue("group_type", TiType.utf8())
        .addValue("group_role_scope", TiType.optional(TiType.utf8()))
        .addValue("group_service_id", TiType.optional(TiType.int64()))
        .addValue("group_parent_id", TiType.optional(TiType.int64()))
        .addValue("group_parent_is_deleted", TiType.optional(TiType.bool()))
        .addValue("group_parent_url", TiType.optional(TiType.utf8()))
        .addValue("group_parent_type", TiType.optional(TiType.utf8()))
        .addValue("group_parent_role_scope", TiType.optional(TiType.utf8()))
        .addValue("group_parent_service_id", TiType.optional(TiType.int64()))
        .addValue("person_id", TiType.int64())
        .addValue("person_is_dismissed", TiType.bool())
        .addValue("person_login", TiType.utf8())
        .addValue("person_is_deleted", TiType.bool())
        .setUniqueKeys(false)
        .build()
    private val abcServicesTableSchema: TableSchema = TableSchema.builder()
        .addValue("id", TiType.int64())
        .addValue("created_at", TiType.timestamp())
        .addValue("slug", TiType.utf8())
        .addValue("state", TiType.utf8())
        .addValue("read_only_state", TiType.optional(TiType.utf8()))
        .addValue("type_id", TiType.int64())
        .addValue("type_code", TiType.utf8())
        .addValue("is_exportable", TiType.bool())
        .setUniqueKeys(false)
        .build()
    private val abcRolesTableSchema: TableSchema = TableSchema.builder()
        .addValue("id", TiType.int64())
        .addValue("created_at", TiType.timestamp())
        .addValue("code", TiType.utf8())
        .addValue("scope_id", TiType.int64())
        .addValue("scope_slug", TiType.utf8())
        .addValue("service_id", TiType.optional(TiType.int64()))
        .addValue("service_slug", TiType.optional(TiType.utf8()))
        .setUniqueKeys(false)
        .build()
    private val abcServiceMembersTableSchema: TableSchema = TableSchema.builder()
        .addValue("id", TiType.int64())
        .addValue("created_at", TiType.timestamp())
        .addValue("modified_at", TiType.timestamp())
        .addValue("person_id", TiType.int64())
        .addValue("person_login", TiType.utf8())
        .addValue("service_id", TiType.int64())
        .addValue("service_slug", TiType.utf8())
        .addValue("role_id", TiType.int64())
        .addValue("role_code", TiType.utf8())
        .addValue("role_scope_id", TiType.int64())
        .addValue("role_scope_slug", TiType.utf8())
        .setUniqueKeys(false)
        .build()

    suspend fun writeIdmGroups(user: String, token: String, groups: Flow<List<IdmGroup>>) {
        val ytClient = YtClient.builder()
            .setCluster(cluster)
            .setRpcCredentials(RpcCredentials(user, token))
            .setRpcOptions(RpcOptions()
                .setStreamingReadTimeout(streamingReadTimeout)
                .setStreamingWriteTimeout(streamingWriteTimeout))
            .build()
        ytClient.use {
            val transaction = ytClient.startTransaction(StartTransaction.master()
                .setTimeout(requestTimeout)).await()
            val success = AtomicBoolean(false)
            try {
                if (transaction.existsNode(idmGroupsTablePath, requestTimeout).await()) {
                    transaction.removeNode(idmGroupsTablePath).await()
                }
                transaction.createNode(idmGroupsTablePath, ObjectType.Table,
                    mapOf(Pair("schema", idmGroupsTableSchema.toYTree())),
                    requestTimeout).await()
                val objectSerializer = YTreeObjectSerializerFactory.forClass(YtIdmGroup::class.java)
                val serializer = MappedRowSerializer.forClass(objectSerializer)
                val writer = transaction.writeTable(WriteTable(YPath.simple(idmGroupsTablePath), serializer)
                    .setTimeout(requestTimeout)).await()
                try {
                    groups.collect { page ->
                        val mapped = page.map { group -> YtIdmGroup(group.id,
                            toTimestamp(group.createdAt), group.slug, group.state, group.type,
                            toTimestamp(group.updatedAt)) }
                        if (!writer.write(mapped)) {
                            writer.readyEvent().await()
                        }
                    }
                } finally {
                    writer.close().await()
                }
                success.set(true)
            } finally {
                if (success.get()) {
                    transaction.commit().await()
                } else {
                    transaction.abort().await()
                }
            }
        }
    }

    suspend fun writeAbcServices(user: String, token: String, groups: Flow<List<AbcService>>) {
        val ytClient = YtClient.builder()
            .setCluster(cluster)
            .setRpcCredentials(RpcCredentials(user, token))
            .setRpcOptions(RpcOptions()
                .setStreamingReadTimeout(streamingReadTimeout)
                .setStreamingWriteTimeout(streamingWriteTimeout))
            .build()
        ytClient.use {
            val transaction = ytClient.startTransaction(StartTransaction.master()
                .setTimeout(requestTimeout)).await()
            val success = AtomicBoolean(false)
            try {
                if (transaction.existsNode(abcServicesTablePath, requestTimeout).await()) {
                    transaction.removeNode(abcServicesTablePath).await()
                }
                transaction.createNode(abcServicesTablePath, ObjectType.Table,
                    mapOf(Pair("schema", abcServicesTableSchema.toYTree())),
                    requestTimeout).await()
                val objectSerializer = YTreeObjectSerializerFactory.forClass(YtAbcService::class.java)
                val serializer = MappedRowSerializer.forClass(objectSerializer)
                val writer = transaction.writeTable(WriteTable(YPath.simple(abcServicesTablePath), serializer)
                    .setTimeout(requestTimeout)).await()
                try {
                    groups.collect { page ->
                        val mapped = page.map { service -> YtAbcService(service.id,
                            toTimestamp(service.createdAt), service.slug, service.state,
                            service.readOnlyState, service.type.id, service.type.code, service.exportable) }
                        if (!writer.write(mapped)) {
                            writer.readyEvent().await()
                        }
                    }
                } finally {
                    writer.close().await()
                }
                success.set(true)
            } finally {
                if (success.get()) {
                    transaction.commit().await()
                } else {
                    transaction.abort().await()
                }
            }
        }
    }

    suspend fun writeAbcRoles(user: String, token: String, groups: Flow<List<AbcRole>>) {
        val ytClient = YtClient.builder()
            .setCluster(cluster)
            .setRpcCredentials(RpcCredentials(user, token))
            .setRpcOptions(RpcOptions()
                .setStreamingReadTimeout(streamingReadTimeout)
                .setStreamingWriteTimeout(streamingWriteTimeout))
            .build()
        ytClient.use {
            val transaction = ytClient.startTransaction(StartTransaction.master()
                .setTimeout(requestTimeout)).await()
            val success = AtomicBoolean(false)
            try {
                if (transaction.existsNode(abcRolesTablePath, requestTimeout).await()) {
                    transaction.removeNode(abcRolesTablePath).await()
                }
                transaction.createNode(abcRolesTablePath, ObjectType.Table,
                    mapOf(Pair("schema", abcRolesTableSchema.toYTree())),
                    requestTimeout).await()
                val objectSerializer = YTreeObjectSerializerFactory.forClass(YtAbcRole::class.java)
                val serializer = MappedRowSerializer.forClass(objectSerializer)
                val writer = transaction.writeTable(WriteTable(YPath.simple(abcRolesTablePath), serializer)
                    .setTimeout(requestTimeout)).await()
                try {
                    groups.collect { page ->
                        val mapped = page.map { role -> YtAbcRole(role.id, role.service?.id, role.service?.slug,
                            role.scope.id, role.scope.slug, role.code,
                            toTimestamp(role.createdAt)) }
                        if (!writer.write(mapped)) {
                            writer.readyEvent().await()
                        }
                    }
                } finally {
                    writer.close().await()
                }
                success.set(true)
            } finally {
                if (success.get()) {
                    transaction.commit().await()
                } else {
                    transaction.abort().await()
                }
            }
        }
    }

    suspend fun writeAbcServiceMembers(user: String, token: String, groups: Flow<List<AbcServiceMember>>) {
        val ytClient = YtClient.builder()
            .setCluster(cluster)
            .setRpcCredentials(RpcCredentials(user, token))
            .setRpcOptions(RpcOptions()
                .setStreamingReadTimeout(streamingReadTimeout)
                .setStreamingWriteTimeout(streamingWriteTimeout))
            .build()
        ytClient.use {
            val transaction = ytClient.startTransaction(StartTransaction.master()
                .setTimeout(requestTimeout)).await()
            val success = AtomicBoolean(false)
            try {
                if (transaction.existsNode(abcServiceMembersTablePath, requestTimeout).await()) {
                    transaction.removeNode(abcServiceMembersTablePath).await()
                }
                transaction.createNode(abcServiceMembersTablePath, ObjectType.Table,
                    mapOf(Pair("schema", abcServiceMembersTableSchema.toYTree())),
                    requestTimeout).await()
                val objectSerializer = YTreeObjectSerializerFactory.forClass(YtAbcServiceMember::class.java)
                val serializer = MappedRowSerializer.forClass(objectSerializer)
                val writer = transaction.writeTable(WriteTable(YPath.simple(abcServiceMembersTablePath), serializer)
                    .setTimeout(requestTimeout))
                    .await()
                try {
                    groups.collect { page ->
                        val mapped = page.map { member -> YtAbcServiceMember(member.id, member.person.id,
                            member.person.login, member.service.id, member.service.slug, member.role.id,
                            member.role.code, member.role.scope.id, member.role.scope.slug,
                            toTimestamp(member.createdAt),
                            toTimestamp(member.modifiedAt)) }
                        if (!writer.write(mapped)) {
                            writer.readyEvent().await()
                        }
                    }
                } finally {
                    writer.close().await()
                }
                success.set(true)
            } finally {
                if (success.get()) {
                    transaction.commit().await()
                } else {
                    transaction.abort().await()
                }
            }
        }
    }

    suspend fun writeStaffGroups(user: String, token: String, groups: List<Flow<List<StaffGroup>>>) {
        val ytClient = YtClient.builder()
            .setCluster(cluster)
            .setRpcCredentials(RpcCredentials(user, token))
            .setRpcOptions(RpcOptions()
                .setStreamingReadTimeout(streamingReadTimeout)
                .setStreamingWriteTimeout(streamingWriteTimeout))
            .build()
        ytClient.use {
            val transaction = ytClient.startTransaction(StartTransaction.master()
                .setTimeout(requestTimeout)).await()
            val success = AtomicBoolean(false)
            try {
                if (transaction.existsNode(staffGroupsTablePath, requestTimeout).await()) {
                    transaction.removeNode(staffGroupsTablePath).await()
                }
                transaction.createNode(staffGroupsTablePath, ObjectType.Table,
                    mapOf(Pair("schema", staffGroupsTableSchema.toYTree())),
                    requestTimeout).await()
                val objectSerializer = YTreeObjectSerializerFactory.forClass(YtStaffGroup::class.java)
                val serializer = MappedRowSerializer.forClass(objectSerializer)
                val writer = transaction.writeTable(WriteTable(YPath.simple(staffGroupsTablePath), serializer)
                    .setTimeout(requestTimeout))
                    .await()
                try {
                    groups.forEach { flow ->
                        flow.collect { page ->
                            val mapped = page.map { group -> YtStaffGroup(group.id, group.deleted, group.url,
                                group.type, group.roleScope, group.service.id, group.parent?.id, group.parent?.deleted,
                                group.parent?.url, group.parent?.type, group.parent?.roleScope,
                                group.parent?.service?.id) }
                            if (!writer.write(mapped)) {
                                writer.readyEvent().await()
                            }
                        }
                    }
                } finally {
                    writer.close().await()
                }
                success.set(true)
            } finally {
                if (success.get()) {
                    transaction.commit().await()
                } else {
                    transaction.abort().await()
                }
            }
        }
    }

    suspend fun writeStaffGroupMemberships(user: String, token: String,
        groups: List<Flow<List<StaffGroupMembership>>>) {
        val ytClient = YtClient.builder()
            .setCluster(cluster)
            .setRpcCredentials(RpcCredentials(user, token))
            .setRpcOptions(RpcOptions()
                .setStreamingReadTimeout(streamingReadTimeout)
                .setStreamingWriteTimeout(streamingWriteTimeout))
            .build()
        ytClient.use {
            val transaction = ytClient.startTransaction(StartTransaction.master()
                .setTimeout(requestTimeout)).await()
            val success = AtomicBoolean(false)
            try {
                if (transaction.existsNode(staffGroupMembershipsTablePath, requestTimeout).await()) {
                    transaction.removeNode(staffGroupMembershipsTablePath).await()
                }
                transaction.createNode(staffGroupMembershipsTablePath, ObjectType.Table,
                    mapOf(Pair("schema", staffGroupMembershipsTableSchema.toYTree())),
                    requestTimeout).await()
                val objectSerializer = YTreeObjectSerializerFactory.forClass(YtStaffGroupMembership::class.java)
                val serializer = MappedRowSerializer.forClass(objectSerializer)
                val writer = transaction.writeTable(WriteTable(YPath.simple(staffGroupMembershipsTablePath), serializer)
                    .setTimeout(requestTimeout))
                    .await()
                try {
                    groups.forEach { flow ->
                        flow.collect { page ->
                            val mapped = page.map { membership -> YtStaffGroupMembership(
                                membership.id,
                                membership.group.id,
                                membership.group.deleted,
                                membership.group.url,
                                membership.group.type,
                                membership.group.roleScope,
                                membership.group.service.id,
                                membership.group.parent?.id,
                                membership.group.parent?.deleted,
                                membership.group.parent?.url,
                                membership.group.parent?.type,
                                membership.group.parent?.roleScope,
                                membership.group.parent?.service?.id,
                                toTimestamp(membership.meta.modifiedAt),
                                membership.person.id,
                                membership.person.official.dismissed,
                                membership.person.login,
                                membership.person.deleted,
                                toTimestamp(membership.joinedAt)) }
                            if (!writer.write(mapped)) {
                                writer.readyEvent().await()
                            }
                        }
                    }
                } finally {
                    writer.close().await()
                }
                success.set(true)
            } finally {
                if (success.get()) {
                    transaction.commit().await()
                } else {
                    transaction.abort().await()
                }
            }
        }
    }

    private fun toTimestamp(value: Instant): UnsignedLong {
        return UnsignedLong.valueOf(TimeUnit.SECONDS.toMicros(value.epochSecond) +
            TimeUnit.NANOSECONDS.toMicros(value.nano.toLong()))
    }

}
