from django.contrib import admin
from django.db.models import F

from intranet.vconf.src.call.models import ConferenceCall, CallTemplate, Node


class CallAdmin(admin.ModelAdmin):
    raw_id_fields = ('template', )

    list_display = (
        'name',
        'stream',
        'record',
        'priority',
        'conf_cms_id',
        'call_cms_id',
        'event_id',
        'event_external_id',
    )

    list_filter = ('state',)

    search_fields = ('name', 'author', 'conf_cms_id', 'call_cms_id', 'event_id', 'event_external_id')


class TemplateAdmin(admin.ModelAdmin):
    list_display = (
        'id',
        'name',
        'stream_description',
        'stream',
        'record',
        'priority',
        'event_id',
        'event_external_id',
    )

    search_fields = ('id', 'name', 'author', 'event_id', 'event_external_id')


class NodeAdmin(admin.ModelAdmin):

    list_display = (
        'id',
        'vm_host',
        'last_update',
        'load_value',
        'load_limit',
        'load_available',
        'percent',
        'is_alive',
        'enabled',
    )

    list_editable = (
        'enabled',
    )

    def is_alive(self, obj):
        return obj.is_alive

    is_alive.boolean = True

    def percent(self, obj):
        return '{:.2f}%'.format(obj.load_value / obj.load_limit * 100)

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        qs = qs.annotate(
            _load_available=F('load_limit') - F('load_value'),
        )
        return qs

    def load_available(self, obj):
        return obj.load_available

    load_available.admin_order_field = '_load_available'


admin.site.register(ConferenceCall, CallAdmin)
admin.site.register(CallTemplate, TemplateAdmin)
admin.site.register(Node, NodeAdmin)
