from datetime import timedelta

from django.db.models import Q
from django.utils import timezone
from django.core.paginator import Paginator, EmptyPage

from intranet.vconf.src.call.models import CallTemplate, ParticipantTemplate, Event
from intranet.vconf.src.call.hydrator import ParticipantsHydrator
from intranet.vconf.src.call.event import get_event_data, get_serialized_event, update_or_create_event


class Template:

    def __init__(self, obj):
        self.obj = obj

    @classmethod
    def create(cls, data):
        obj = CallTemplate()
        self = cls(obj=obj)
        self.update(data=data)
        return self

    @classmethod
    def find(cls, for_user, obj_id=None):
        """Найти шаблоны пользователя"""
        qs = (
            CallTemplate.objects
            .filter(owners__contains=[for_user.login])
            .order_by('-id')
        )
        if obj_id:
            qs = qs.filter(id=obj_id)
        return [cls(obj=obj) for obj in qs]

    @classmethod
    def find_relevant(cls, for_user, event_external_id=None):
        """
        Вынес это в отдельный метод, потому что метод find скорее подразумевает,
        что там будет логика по И, а нам тут надо ИЛИ
        """
        query = Q(owners__contains=[for_user.login])
        if event_external_id is not None:
            query |= Q(event_external_id=event_external_id)

        qs = CallTemplate.objects.filter(query).order_by('-id')
        return [cls(obj=obj) for obj in qs]

    @classmethod
    def find_streams(cls, page=None, limit=None):
        """Метод используется для получения шаблонов с предстоящими трансляциями"""
        qs = (
            CallTemplate.objects
            .filter(stream=True, next_event__isnull=False)
            .filter(next_event__start_time__gte=timezone.now())
            .order_by('-priority', '-next_event__start_time')
        )

        if page is not None and limit is not None:
            try:
                qs = Paginator(qs, limit).page(page)
            except EmptyPage:
                return []

        return [cls(obj=obj) for obj in qs]

    def update_event(self, data):
        try:
            event = Event.objects.get(id=data['master_id'])
        except Event.DoesNotExist:
            event = Event()
            event.id = data['master_id']

        event.external_id = data['externalId']
        event.start_time = data['startTs']
        event.end_time = data['endTs']
        event.name = data['name']
        event.description = data['description']
        self.obj.next_event = event
        event.save()

    def update(self, data):
        self.obj.name = data['name']
        self.obj.duration = timedelta(minutes=data['duration'])
        self.obj.event_id = data.get('event_id')
        self.obj.event_external_id = data.get('event_external_id')
        self.obj.stream_description = data.get('stream_description', '')
        self.obj.stream_picture = data.get('stream_picture', '')
        self.obj.stream = data['stream']
        self.obj.record = data['record']
        self.obj.owners = data['owners']

        self.obj.next_event = update_or_create_event(data['event']) if data.get('event') else None
        self.obj.save()

        if self.obj.id is None:
            cur_participants = {}
        else:
            cur_participants = {(pt.obj_id, pt.obj_type): pt for pt in (
                ParticipantTemplate.objects
                .filter(call_template=self.obj)
            )}

        for pt_data in data['participants']:
            pt = cur_participants.pop(
                (pt_data['id'], pt_data['type']),
                ParticipantTemplate()
            )
            pt.call_template = self.obj
            pt.obj_id = pt_data['id']
            pt.obj_type = pt_data['type']
            pt.method = pt_data['method']
            pt.save()

        (
            ParticipantTemplate.objects
            .filter(id__in=[pt.id for pt in cur_participants.values()])
            .delete()
        )

    def delete(self):
        self.obj.delete()

    def as_brief_dict(self, tz=None):
        item = {
            'id': self.obj.id,
            'name': self.obj.name,
            'priority': self.obj.priority,
            'stream_description': self.obj.stream_description,
            'stream_picture': self.obj.stream_picture,
        }

        if self.obj.next_event:
            item['next_event'] = get_serialized_event(event=self.obj.next_event, tz=tz)

        return item

    def as_dict(self, lang=None, hydrator=None):
        assert lang or hydrator
        participants = (
            ParticipantTemplate.objects
            .filter(call_template=self.obj)
        )
        participants = [
            {
                'id': p.obj_id,
                'type': p.obj_type,
                'method': p.method,
            } for p in participants
        ]

        if hydrator is None:
            hydrator = ParticipantsHydrator(lang=lang)

        hydrator.add_to_fetch(participants)

        data = self.as_brief_dict()
        data.update({
            'duration': self.obj.duration.seconds // 60,
            'event_id': self.obj.event_id,
            'event_external_id': self.obj.event_external_id,
            'stream': self.obj.stream,
            'record': self.obj.record,
            'owners': self.obj.owners,
            'participants': (
                hydrator.hydrate(pt).data
                for pt in participants
            )
        })
        return data

    def as_detail_dict(self, user):
        data = self.as_dict(lang=user.lang)
        data['event'] = get_event_data(data['event_id'], data['event_external_id'], user)
        return data
