import re

from django.conf import settings

from intranet.vconf.src.call.models import Event
from intranet.vconf.src.ext_api.calendar import get_event_info, CalendarError
import pytz

rgx = re.compile(r'%s\S+' % settings.ZOOM_URL)


def get_serialized_event(event, tz):
    start_time = event.start_time if not tz else event.start_time.astimezone(pytz.timezone(tz))
    end_time = event.end_time if not tz else event.end_time.astimezone(pytz.timezone(tz))
    return {
        'id': event.id,
        'start_time': start_time,
        'end_time': end_time,
        'description': event.description,
    }


def get_event_data(event_id=None, event_external_id=None, user=None):
    event_data = {}
    if event_id or event_external_id:
        if event_id:
            event_external_id = None
        try:
            event = get_event_info(event_id, event_external_id, user)
        except CalendarError:
            event = None

        if event:
            event_data = {
                'id': event_id,
                'external_id': event_external_id,
                'name': event['name'],
                'startTs': event['startTs']
            }

    return event_data


def user_is_event_participant(user, event):
    if not event:
        return False
    logins = [a['login'] for a in event['attendees'] if 'login' in a]
    organizer = event.get('organizer', {})
    if 'login' in organizer:
        logins.append(organizer['login'])
    return user.login in logins


def update_or_create_event(data, regenerate_secret=False):
    try:
        event = Event.objects.get(id=data['master_id'])
        if regenerate_secret or not event.secret:
            event.generate_secret()
    except Event.DoesNotExist:
        event = Event()
        event.id = data['master_id']
        event.generate_secret()

    event.external_id = data['externalId']
    event.start_time = data['startTs']
    event.end_time = data['endTs']
    event.name = data['name']
    event.description = data['description']
    event.save()
    return event


def find_zoom_link(description):
    match = rgx.search(description)
    if match:
        return match[0]
