import logging

from intranet.vconf.src.ext_api.staff import get_person_info, get_room_info, get_room_info_by_slugs
from intranet.vconf.src.call.models import PARTICIPANT_TYPES, CALL_METHODS
from intranet.vconf.src.call.participant_ctl import (
    PersonParticipant,
    RoomParticipant,
    ArbitraryParticipant,
    CmsParticipant,
)
from intranet.vconf.src.call.manager import CallManager
from intranet.vconf.src.call.models import ConferenceCall, UserSettings
from intranet.vconf.src.call.event import find_zoom_link


logger = logging.getLogger(__name__)


def get_slug(resource):
    return resource['email'].split('@')[0]


class Hydrator:
    """
    Класс нужен, чтобы обогощать какие-либо сущности данными из API пачкой
    """
    def __init__(self):
        self.need_fetch = False

    def add_to_fetch(self, items):
        """
        Добавляет `items` в список того, что нужно гидрировать
        Пока не вызван метод hydrate, никаких запросов в API не происходит
        :return:
        """
        for item in items:
            self.perform_add(item)
        self.need_fetch = True

    def perform_add(self, item):
        raise NotImplementedError

    def fetch(self):
        """
        Достает все данные одной пачкой из API
        """
        self.perform_fetch()
        self.need_fetch = False
        self.clean_after_fetch()

    def clean_after_fetch(self):
        raise NotImplementedError

    def perform_fetch(self):
        raise NotImplementedError

    def hydrate(self, item):
        """
        Гидрирует один элемент
        """
        if self.need_fetch:
            self.fetch()
        return self.perform_hydrate(item)

    def perform_hydrate(self, item):
        raise NotImplementedError


class HydratedEvent:

    def __init__(self, event, call, rooms):
        self.event = event
        self.call = call
        self.rooms = rooms

    def as_dict(self, include_zoom=False):
        event = dict(self.event)
        if self.call:
            event['call'] = self.call.as_brief_dict()
        else:
            event['call'] = None
        event['rooms'] = [room.data for room in self.rooms]
        event['zoom_link'] = find_zoom_link(event.get('description', ''))
        if event['zoom_link'] and include_zoom:
            event['zoom'] = ArbitraryParticipant.from_raw(
                raw_data={
                    'id': event['zoom_link'],
                    'type': PARTICIPANT_TYPES.arbitrary,
                },
                safe=True,
            ).data
        return event


class EventsHydrator(Hydrator):

    def __init__(self, user):
        super().__init__()

        self.user = user

        self.events_ids = []
        self.rooms_slugs = []

        self.calls_data = {}
        self.rooms_data = {}

    def perform_add(self, item):
        self.events_ids.append(item['id'])
        for resource in item.get('resources', []):
            resource['slug'] = get_slug(resource)
            self.rooms_slugs.append(resource['slug'])

    def clean_after_fetch(self):
        self.events_ids = []
        self.rooms_slugs = []

    def perform_fetch(self):
        rooms_from_staff = get_room_info_by_slugs(self.rooms_slugs)
        self.rooms_data.update({
            p['name']['exchange']: p for p in rooms_from_staff['result']
        })

        active_calls = (
            ConferenceCall.objects
            .filter(
                event_id__in=self.events_ids,
                state=ConferenceCall.STATES.active,
            )
        )
        self.calls_data.update({
            call.event_id: call for call in active_calls
        })

    def _hydrate_rooms(self, event):
        rooms = []
        for resource in event.get('resources', []):
            if resource['resourceType'] not in ('room', 'protected-room'):
                continue

            slug = get_slug(resource)
            if slug not in self.rooms_data:
                logger.error('Unknown room slug `%s` in event %s', slug, event['id'])
                continue

            ext_data = self.rooms_data[slug]
            raw_data = {
                'id': ext_data['id'],
                'type': PARTICIPANT_TYPES.room,
                'method': CALL_METHODS.cisco,
            }
            rooms.append(
                RoomParticipant.from_raw(
                    raw_data=raw_data,
                    ext_data=ext_data,
                    lang=self.user.lang,
                )
            )
        return rooms

    def perform_hydrate(self, item):
        call = None
        if item['id'] in self.calls_data:
            call = CallManager(
                obj=self.calls_data[item['id']],
                for_user=self.user,
                event=item,
            )

        return HydratedEvent(
            event=item,
            call=call,
            rooms=self._hydrate_rooms(item),
        )


class ParticipantsHydrator(Hydrator):

    def __init__(self, lang, safe=False):
        """
        :param lang: язык пользователя
        :param safe: если True, исключения не будут рейзиться
        """
        super().__init__()
        self.lang = lang
        self.rooms_ids = []
        self.persons_ids = []
        self.rooms_data = {}
        self.persons_data = {}
        self.safe = safe

    def perform_add(self, item):
        if item['type'] == PARTICIPANT_TYPES.room:
            self.rooms_ids.append(str(item['id']))
        elif item['type'] == PARTICIPANT_TYPES.person:
            self.persons_ids.append(item['id'])

    def perform_fetch(self):
        if self.rooms_ids:
            rooms_data = get_room_info(self.rooms_ids)
            self.rooms_data.update({str(p['id']): p for p in rooms_data['result']})

        if self.persons_ids:
            persons_data = get_person_info(self.persons_ids)
            default_method_map = dict(
                UserSettings.objects
                .filter(login__in=self.persons_ids)
                .values_list('login', 'method')
            )
            for person in persons_data['result']:
                login = person['login']
                if login in default_method_map:
                    person['settings'] = {
                        'method': default_method_map[login],
                    }
                self.persons_data[login] = person

    def clean_after_fetch(self):
        self.rooms_ids = []
        self.persons_ids = []

    def perform_hydrate(self, item):
        if item['type'] == PARTICIPANT_TYPES.room:
            result = RoomParticipant.from_raw(
                raw_data=item,
                ext_data=self.rooms_data.get(str(item['id']), {}),
                lang=self.lang,
            )
        elif item['type'] == PARTICIPANT_TYPES.person:
            result = PersonParticipant.from_raw(
                raw_data=item,
                ext_data=self.persons_data.get(item['id']),
                lang=self.lang,
            )
        elif item['type'] == PARTICIPANT_TYPES.arbitrary:
            result = ArbitraryParticipant.from_raw(item, safe=self.safe)
        elif item['type'] == PARTICIPANT_TYPES.cms:
            result = CmsParticipant.from_raw(item)
        else:
            raise TypeError

        return result
