import random
import logging
from datetime import timedelta, datetime
from functools import wraps
from typing import Optional
from uuid import uuid4

import pytz
import ylock
import yenv

from django.contrib.auth import get_user_model
from django.utils import timezone
from django.db.models import Count, Q
from django.core.paginator import Paginator, EmptyPage
from django.conf import settings

from intranet.vconf.src.call.models import ConferenceCall, Participant, Record, CallTemplate, CALL_METHODS
from intranet.vconf.src.call.event import get_serialized_event
from intranet.vconf.src.call.retranslator import get_retranslator_host, get_ether_url
from intranet.vconf.src.call.constants import CallFilter, STREAM_LANGUAGES
from intranet.vconf.src.call.event import user_is_event_participant, update_or_create_event
from intranet.vconf.src.ext_api.cms import CMSApi
from intranet.vconf.src.ext_api.streaming import start_stream, stop_stream
from intranet.vconf.src.ext_api.messenger import MessengerAPI, MessengerAPIError
from intranet.vconf.src.ext_api.calendar import get_next_event, CalendarError

from intranet.vconf.src.call.ether import get_ether_id
from intranet.vconf.src.call.models import CALL_STATES

lock_manager = ylock.backends.create_manager(**settings.YLOCK)
log = logging.getLogger(__name__)
User = get_user_model()


def permission_required(permission_method):
    def decorator(func):
        @wraps(func)
        def wrapper(self, *args, **kwargs):
            if not getattr(self, permission_method)():
                raise self.PermissionDenied
            return func(self, *args, **kwargs)

        return wrapper

    return decorator


class CallManager:
    class Error(Exception):
        pass

    class CallAlreadyExists(Error):
        pass

    class ParticipantNotFound(Error):
        pass

    class PermissionDenied(Error):
        def __init__(self, message=None):
            super().__init__(message or 'permission denied')

    def __init__(self, obj: ConferenceCall, for_user: Optional[User] = None, event: Optional[dict] = None):
        self.obj = obj
        self.for_user = for_user
        self.event = event
        if self.event:
            assert self.obj.event_id == self.event['id']
        self.cms = CMSApi(obj.cms_node)

    @staticmethod
    def _generate_id():
        return str(datetime.timestamp(datetime.now())).split('.')[1] + str(random.randrange(1000, 9999))

    def _fetch_event_external_id(self) -> None:
        """
        Получаем event_external_id из шаблона или события в Календаре.

        TODO: вынести отсюда валидацию в формы
        TODO: Разобраться с тем, что автор может быть None
        """
        assert self.for_user or not self.obj.event_id, 'The author is required'
        template = None
        if self.obj.template_id:
            try:
                template = CallTemplate.objects.get(id=self.obj.template_id)
                self.obj.priority = template.priority
                self.obj.event_external_id = template.event_external_id
            except CallTemplate.DoesNotExist:
                raise self.Error('Template `%d` was not found' % self.obj.template_id)

        event = None
        if self.obj.event_id:
            try:
                event = get_next_event(event_id=self.obj.event_id, user=self.for_user)
                self.obj.next_event = update_or_create_event(event)
                self.obj.event_external_id = event['externalId']
                self.obj.secret = self.obj.next_event.secret
            except CalendarError:
                raise self.Error('Calendar error for event `%s`' % self.obj.event_id)

        if template and event and template.event_external_id != event['externalId']:
            raise self.Error('event_external_id conflict')

    @classmethod
    def create(cls, data: dict, author: Optional[User] = None, master_call: Optional[ConferenceCall] = None):
        obj = ConferenceCall()
        obj.meeting_id = cls._generate_id()
        obj.secret = str(uuid4())
        obj.uri = obj.meeting_id + '_conf'
        # звонки с названием длиннее 109 символов не создаются в ручке cisco coSpaces
        obj.name = data.get('name', obj.uri)[:109]
        obj.event_id = data.get('event_id')
        obj.template_id = data.get('template_id')
        obj.duration = timedelta(minutes=data['duration'])
        obj.stream = data['stream']
        obj.record = data['record']
        obj.is_private_stream = data.get('is_private_stream', False)
        obj.author_login = data.get('author_login') if author is None else author.login
        obj.cms_node = CMSApi.get_node().api_url

        obj.lang = obj.template.lang if obj.template else STREAM_LANGUAGES.ru
        obj.master_call = master_call

        instance = cls(obj=obj, for_user=author)
        instance._fetch_event_external_id()
        participants = data['participants']

        if instance.obj.event_external_id:
            lock_name = f'call_create.{obj.event_external_id}'
            with lock_manager.lock(lock_name, block=True, block_timeout=8, timeout=5):
                instance._create(participants)
        else:
            instance._create(participants)

        if obj.template_id and obj.template.master_template is None:
            instance._create_related_calls()

        return instance

    def _create(self, participants):

        self._check_another_call_exists()

        if self.obj.stream:
            self._init_stream()

        try:
            self._create_space()
            self._create_call()
        except self.Error as e:
            log.exception(e)
        else:
            if self.obj.stream:
                start_stream(self._stream_id)
                self._create_chat()

        one = False
        for participant in participants:
            try:
                self.add_participant(participant)
            except self.Error as e:
                log.exception(e)
            else:
                if not one:
                    self._start_call()
                one = True

        if not one:
            log.error('call without participants')
            raise self.Error('Unexpected error in CallManager.create')

    def _check_another_call_exists(self):
        if self.obj.event_external_id:
            another_call = (
                ConferenceCall.objects
                .filter(
                    state=ConferenceCall.STATES.active,
                    event_external_id=self.obj.event_external_id,
                )
                .exists()
            )
            if another_call:
                raise self.CallAlreadyExists(
                    'There is already active call with event_id %s' % self.obj.event_id
                )

    def _create_related_calls(self):
        """
        Запустить звонки, для которыx этот звонок является родительским (master_call)
        """

        from intranet.vconf.src.lib.utils import hydrate_participants
        from intranet.vconf.src.call.call_template import Template

        master_template_id = self.obj.template_id
        related_templates = CallTemplate.objects.filter(master_template_id=master_template_id)
        robot = User.objects.get(username=settings.VCONF_ROBOT_LOGIN)

        for template in related_templates:
            template_dict = Template(template).as_dict(lang=template.lang)
            template_dict.update({
                'record': template.record,
                'stream': template.stream,
                'is_private_stream': True,
                'template_id': template.id,
            })
            template_dict['participants'] = hydrate_participants(list(template_dict['participants']))
            CallManager.create(
                data=template_dict,
                author=robot,
                master_call=self.obj,
            )

    @classmethod
    def find_calls(cls, for_user, call_filter: CallFilter, base_qs=None):
        """
        :param for_user: :type vconf.users.models.User
        :param base_qs: Базовый Queryset, например, чтобы прокинуть prefetch_related там, где надо
        TODO: Чтобы не было путаницы, лучше show_all параметр выпилить и по дефолту отдавать все.
        А для своих звонков придумать другой параметр.
        """
        if base_qs is not None:
            qs = base_qs
        else:
            qs = ConferenceCall.objects.all()

        qs = qs.select_related('master_call')

        if call_filter.sort:
            qs = qs.order_by(call_filter.sort, '-start_time')
        else:
            qs = qs.order_by('-start_time')

        relevant_calls_query = (
            Q(conf_cms_id__in=(
                Participant.objects
                .filter(obj_type='person', obj_id=for_user.login)
                .values('conf_call')
            ))
            | Q(author_login=for_user.login)
        )

        if hasattr(for_user, 'call_secret'):
            relevant_calls_query |= Q(secret=for_user.call_secret)

        if call_filter.show_all and for_user.can_see_streams:
            if not for_user.is_admin:
                qs = qs.filter(relevant_calls_query | Q(stream=True))
        else:
            qs = qs.filter(relevant_calls_query)

        if call_filter.state:
            qs = qs.filter(state=call_filter.state)

        if call_filter.with_record is not None:
            qs = qs.filter(record=call_filter.with_record)

        if call_filter.with_stream is not None:
            qs = qs.filter(stream=call_filter.with_stream)

        if call_filter.with_active_participants:
            qs = qs.filter(
                conf_cms_id__in=(
                    Participant.objects
                    .filter(state=Participant.STATES.active)
                    .values('conf_call__conf_cms_id')
                )
            )

        if call_filter.conf_cms_id:
            qs = qs.filter(conf_cms_id=call_filter.conf_cms_id)

        if call_filter.template_id:
            qs = qs.filter(template_id=call_filter.template_id).select_related('template')

        if call_filter.event_external_id:
            qs = qs.filter(event_external_id=call_filter.event_external_id)

        if call_filter.page is not None and call_filter.limit is not None:
            try:
                qs = Paginator(qs, call_filter.limit).page(call_filter.page)
            except EmptyPage:
                return []

        return [cls(obj=call, for_user=for_user) for call in qs]

    @classmethod
    def find_active(cls, for_user, conf_cms_id=None):
        return cls.find_calls(
            for_user=for_user,
            call_filter=CallFilter(
                state=CALL_STATES.active,
                conf_cms_id=conf_cms_id,
                show_all=True,
            ),
        )

    @classmethod
    def find_ended(cls, for_user, conf_cms_id=None):
        return cls.find_calls(
            for_user=for_user,
            call_filter=CallFilter(
                state=CALL_STATES.ended,
                conf_cms_id=conf_cms_id,
            ),
        )

    @classmethod
    def find_outdated(cls, for_user=None):
        now = timezone.now()
        qs = (
            ConferenceCall.objects
            .filter(
                Q(stop_time__lte=now, state=ConferenceCall.STATES.active)
                | Q(state=ConferenceCall.STATES.broken)
                | Q(state=ConferenceCall.STATES.ended_in_cms)
            )
        )
        return [cls(obj=call, for_user=for_user) for call in qs]

    @classmethod
    def get_by_id(cls, conf_cms_id=None, call_cms_id=None, for_user=None):
        if conf_cms_id:
            obj = ConferenceCall.objects.get(conf_cms_id=conf_cms_id)
        else:
            obj = ConferenceCall.objects.get(call_cms_id=call_cms_id)
        return cls(obj=obj, for_user=for_user)

    @classmethod
    def get_active_by_event_id(cls, event_id: int, for_user: User = None):
        obj = (
            ConferenceCall.objects
            .filter(event_id=event_id, state=CALL_STATES.active)
            .first()
        )
        if obj is None:
            return None
        return cls(obj=obj, for_user=for_user)

    @property
    def invite_link(self):
        if yenv.type == 'production':
            invite_link_template = 'https://cms.yandex-team.ru/meeting/{mid}?secret={secret}'
        else:
            invite_link_template = 'https://test.cms.yandex-team.ru/meeting/{mid}?secret={secret}'
        return invite_link_template.format(secret=self.obj.secret, mid=self.obj.meeting_id, )

    def _create_space(self):
        owner = CMSApi.get_free_owner()
        res = self.cms.space_create(
            call_id=self.obj.meeting_id,
            uri=self.obj.uri,
            name=self.obj.name,
            secret=self.obj.secret,
            stream=self.obj.stream,
            record=self.obj.record,
            stream_id=self._stream_id,
            owner=owner,
        )
        if res.status_code != 200:
            raise self.Error('bad request ' + str(res.status_code) + res.text)
        self.obj.conf_cms_id = res.headers['Location'][17:]
        log.info('Conf %s created on node %s with owner %s', self.obj.conf_cms_id, self.obj.cms_node, owner)
        self.obj.save()

    def _create_call(self):
        res = self.cms.call_create(
            self.obj.conf_cms_id, self.obj.uri
        )
        if res.status_code != 200:
            raise self.Error('bad request ' + str(res.status_code) + res.text)
        self.obj.call_cms_id = res.headers['Location'][14:]
        log.info('Call %s created in conf %s', self.obj.call_cms_id, self.obj.conf_cms_id)
        self.obj.save()

    @property
    def _stream_id(self) -> str:
        return self.obj.ether_back_id or self.obj.uri

    def _init_stream(self) -> None:
        ether_id = get_ether_id(call=self.obj)
        self.obj.ether_back_id = ether_id.back_id
        self.obj.ether_front_id = ether_id.front_id
        log.info('EtherId %s set to conf %s', ether_id, self.obj.conf_cms_id)
        self.obj.save()

    def _create_chat(self):
        try:
            data = MessengerAPI.create_chat(
                name=self.obj.name,
                description=self.obj.name,
                admin_logins=[self.obj.author_login],
                member_logins=[self.obj.author_login],
                chat_id=self.obj.uri,
                is_hural=self.obj.template_id == settings.HURAL_TEMPLATE_ID,
            )
        except MessengerAPIError:
            log.error('Failed to create chat. Conference call `%d` without chat!', self.obj.id)
        else:
            self.obj.chat = True
            self.obj.has_active_chat = True
            self.obj.chat_invite_hash = data['invite_hash']
            log.info('Chat %s created in conf %s', self.obj.chat_id, self.obj.conf_cms_id)
            self.obj.save()

    def _start_call(self):
        self.obj.state = ConferenceCall.STATES.active
        self.obj.start_time = timezone.now()
        self.obj.stop_time = self.obj.start_time + self.obj.duration
        self.obj.save()

    @permission_required('can_user_write')
    def add_participant(self, participant):
        participant.add_to_call(self)

    def toggle_microphone(self, participant):
        self._toggle_participant(participant, 'microphone_active')

    def toggle_camera(self, participant):
        self._toggle_participant(participant, 'camera_active')

    @permission_required('can_user_write')
    def _toggle_participant(self, participant, attr):
        pt = (
            Participant.objects
            .get(
                obj_id=participant['id'],
                obj_type=participant['type'],
                conf_call=self.obj
            )
        )

        setattr(pt, attr, not getattr(pt, attr))

        actions = {
            'cam': pt.camera_active,
            'mic': pt.microphone_active,
        }

        res = self.cms.participant_mute(pt.cms_id, actions)
        if res.status_code != 200:
            raise self.Error('bad request ' + str(res.status_code) + res.text)
        elif 'total="0"' in res.text:
            raise self.Error('Call legs for participant {} was not found'.format(pt.cms_id))
        log.info(
            'The participant %s id==%s was changed',
            pt.obj_id, pt.obj_id
        )
        pt.save()

    @permission_required('can_user_write')
    def remove_participant(self, participant, only_disconnect):
        try:
            participant = (
                Participant.objects
                .get(
                    obj_id=participant['id'],
                    obj_type=participant['type'],
                    conf_call=self.obj
                )
            )
        except Participant.DoesNotExist:
            log.error(
                'Participant id==%s with type %s does not exist in call %s',
                participant['id'], participant['type'], self.obj.conf_cms_id
            )
            raise self.ParticipantNotFound(participant['id'])

        if participant.method != CALL_METHODS.email:
            res = self.cms.participant_delete(participant.cms_id)
            if not (res.status_code == 200 or 'participantDoesNotExist' in res.text):
                raise self.Error('bad request ' + str(res.status_code) + ' ' + res.text)
        log.info(
            'The participant %s id==%s is removed from the call %s',
            participant.obj_id, participant.obj_id, self.obj.conf_cms_id
        )
        if only_disconnect:
            participant.state = Participant.STATES.disconnected
        else:
            participant.state = Participant.STATES.ended
        participant.save()
        self._stop_if_empty()

    def _stop_if_empty(self):
        active_part_qs = (
            Participant.objects
            .filter(conf_call=self.obj)
            .exclude(state=Participant.STATES.ended)
        )
        if not active_part_qs.exists():
            self.stop()

    @permission_required('can_user_write')
    def stop(self):
        if self.obj.state == ConferenceCall.STATES.ended:
            log.warning(
                'The conference %s is already ended', self.obj.conf_cms_id
            )
        elif self.obj.state != ConferenceCall.STATES.ended_in_cms:
            res = self.cms.space_delete(self.obj.conf_cms_id)
            if 'coSpaceDoesNotExist' in res.text:
                log.warning('The conference %s does not exist on CMS', self.obj.conf_cms_id)
            elif res.status_code != 200:
                raise self.Error('bad request ' + str(res.status_code) + res.text)
            else:
                log.info('The conference %s successfully ended', self.obj.conf_cms_id)

        if self.obj.stream:
            stop_stream(self._stream_id)

        (
            Participant.objects
            .filter(conf_call=self.obj)
            .update(state=Participant.STATES.ended)
        )
        self.obj.state = ConferenceCall.STATES.ended
        self.obj.stop_time = timezone.now()
        self.obj.save()

    def _get_participant_logins(self):
        return [
            p.obj_id
            for p in self.obj.participants.all()
            if p.obj_type == 'person'
        ]

    def can_user_write(self):
        return (
            self.for_user is not None
            and (
                self.for_user.is_admin
                or getattr(self.for_user, 'call_secret', None) == self.obj.secret
                or self.for_user.login == self.obj.author_login
                or self.for_user.login in self._get_participant_logins()
                or user_is_event_participant(self.for_user, self.event)
            )
        )

    def can_user_download_record(self):
        return self.can_user_write() and not self.for_user.is_ip_external

    def _get_serialized_participants(self, hydrator=None, lang=None):
        from intranet.vconf.src.call.hydrator import ParticipantsHydrator

        participants = [
            {
                'id': p.obj_id,
                'type': p.obj_type,
                'number': p.number,
                'camera': p.camera,
                'microphone': p.microphone,
                'camera_active': p.camera_active,
                'microphone_active': p.microphone_active,
                'method': p.method,
                'state': p.state,
            } for p in self.obj.participants.all()
        ]

        if hydrator is None:
            hydrator = ParticipantsHydrator(lang=lang, safe=True)

        hydrator.add_to_fetch(participants)
        return (hydrator.hydrate(pt).data for pt in participants)

    def as_brief_dict(self):
        data = {
            'id': self.obj.conf_cms_id,
            'uri': self.obj.uri,
            'name': self.obj.name,
            'state': self.obj.state,
            'stream': self.obj.stream,
            'record': self.obj.record,
        }
        if self.can_user_write():
            data['secret'] = self.obj.secret
            data['invite_link'] = self.invite_link
        return data

    def as_dict(self, tz=None, lang=None, hydrator=None):
        """
        Базовый сериализатор звонка, можно использовать в списковых ручках
        """
        tz = tz or self.for_user and self.for_user.tz
        lang = lang or self.for_user and self.for_user.lang
        assert lang or hydrator
        assert tz

        data = self.as_brief_dict()

        if self.obj.start_time:
            start_time = self.obj.start_time.astimezone(pytz.timezone(tz))
            stop_time = start_time + self.obj.duration
        else:
            start_time = None
            stop_time = None

        data.update({
            'stream_description': self.obj.template.stream_description if self.obj.template_id else '',
            'stream_picture': self.obj.template.stream_picture if self.obj.template_id else '',
            'template_id': self.obj.template_id,
            'event_id': self.obj.event_id,
            'event_external_id': self.obj.event_external_id,

            'chat': self.obj.master_call.chat if self.obj.master_call else self.obj.chat,
            'chat_id': self.obj.master_call.chat_id if self.obj.master_call else self.obj.chat_id,
            'chat_invite_hash': (
                self.obj.master_call.chat_invite_hash
                if self.obj.master_call
                else self.obj.chat_invite_hash
            ),

            'stream_id': self._stream_id,

            'start_time': start_time,
            'duration': self.obj.duration.seconds // 60,
            'stop_time': stop_time,

            'author_login': self.obj.author_login,
            'participants': self._get_serialized_participants(hydrator, lang),

            'priority': self.obj.priority,

            'lang': self.obj.lang,
        })

        if self.obj.template and self.obj.template.next_event:
            data['event'] = get_serialized_event(event=self.obj.template.next_event, tz=tz)

        if self.obj.stream:
            data.update({
                'ether_url': get_ether_url(self.obj),
                'ether_back_id': self.obj.ether_back_id,
                'ether_front_id': self.obj.ether_front_id,
                'viewers_count': self.obj.viewers_count,
            })

        return data

    def as_detail_dict(self, tz=None, lang=None, hydrator=None):
        """
        Сериализатор звонка, который не стоит использовать в цикле,
        потому что на каждый звонок будет запрос по сети
        """
        data = self.as_dict(tz=tz, lang=lang, hydrator=hydrator)
        data['retranslator_host'] = get_retranslator_host(self.for_user)

        if self.obj.stream:
            related_calls = CallManager.find_related_calls(self.obj)
            if related_calls.count() > 1:
                calls_bundle = {
                    call.lang: call.conf_cms_id
                    for call in related_calls
                }
                data['translated_streams'] = calls_bundle

        return data

    @permission_required('can_user_write')
    def update_duration(self, duration):
        self.obj.duration = timedelta(minutes=duration)
        self.obj.stop_time = self.obj.start_time + self.obj.duration
        self.obj.save()

    def add_record(self, data):
        Record.objects.get_or_create(
            conf_call=self.obj,
            file_name=data['name'],
            node=data['node'],
        )

    @permission_required('can_user_download_record')
    def get_record(self):
        record = Record.objects.filter(
            conf_call__conf_cms_id=self.obj.conf_cms_id,
            is_deleted=False,
        )
        if record:
            return record[0]

    @permission_required('can_user_download_record')
    def delete_record(self):
        record = Record.objects.get(conf_call__conf_cms_id=self.obj.conf_cms_id)
        record.conf_call.record = False
        record.is_deleted = True
        record.conf_call.save()
        record.save()

    @staticmethod
    def find_active_with_participant_count():
        active_participants_count = Count(
            expression='participants',
            filter=Q(participants__state=Participant.STATES.active),
        )
        return list(
            ConferenceCall.objects
            .filter(state=ConferenceCall.STATES.active)
            .annotate(active_participants_count=active_participants_count)
            .values('event_id', 'event_external_id', 'active_participants_count')
        )

    @classmethod
    def find_related_calls(cls, call: ConferenceCall):
        """
        Звонки с общим родительским звонком и родительский звонок
        Такие звонки запускаются одновременно при запуске родительского шаблона
        и переключение языка переключает пользователя между этими звонками
        """
        master_call = call.master_call or call
        return (
            ConferenceCall.objects
            .filter(state=CALL_STATES.active)
            .filter(Q(conf_cms_id=master_call.conf_cms_id) | Q(master_call=master_call))
        )
