from itertools import chain
from dateutil.parser import parse

from django.db.models import ObjectDoesNotExist

from intranet.vconf.src.call.call_template import Template
from intranet.vconf.src.call.hydrator import ParticipantsHydrator, EventsHydrator
from intranet.vconf.src.ext_api.calendar import get_event_info
from intranet.vconf.src.lib.utils import round_duration


class EventConflict(Exception):
    pass


class CallDataMerger:
    """
    Собирает данные для создания звонка из разных источников (Календарь, Шаблон)
    """
    def __init__(self, user, event_id=None, template_id=None):
        self.user = user
        self.event_id = event_id
        self.template_id = template_id
        self.participants_hydrator = ParticipantsHydrator(lang=self.user.lang)
        self._fetch_event_and_template()

    def _get_one_template_or_nothing(self):
        event_templates = [
            template for template in self.relevant_templates
            if template.obj.event_external_id == self.event_external_id
        ]
        if len(event_templates) != 1:
            return
        template = event_templates[0]
        if self.user.login in template.obj.owners:
            return template

    def _get_relevant_templates(self):
        return (
            Template.find_relevant(
                for_user=self.user,
                event_external_id=self.event_external_id,
            )
        )

    def _get_hydrated_event(self):
        if not self.event:
            return
        hydrator = EventsHydrator(user=self.user)
        hydrator.add_to_fetch([self.event])
        return hydrator.hydrate(self.event).as_dict(include_zoom=True)

    def _get_hydrated_template(self):
        if not self.template:
            return
        return self.template.as_dict(
            lang=self.user.lang,
            hydrator=self.participants_hydrator,
        )

    def _fetch_event(self, event_id):
        self.event = get_event_info(event_id=event_id, user=self.user)
        self.event_id = self.event['id']
        self.event_external_id = self.event['externalId']

    def _fetch_event_and_template(self):
        self.event = None
        self.template = None
        self.event_external_id = None

        if self.event_id:
            self._fetch_event(self.event_id)

        self.relevant_templates = self._get_relevant_templates()
        relevant_templates_map = {template.obj.id: template for template in self.relevant_templates}

        if self.template_id:
            if self.template_id not in relevant_templates_map:
                raise ObjectDoesNotExist('Template does not exist')

            self.template = relevant_templates_map[self.template_id]
            if self.template.obj.event_external_id:
                if self.event:
                    if self.template.obj.event_external_id != self.event_external_id:
                        raise EventConflict
                else:
                    self.event = self._fetch_event(self.template.obj.event_id)
        elif self.event_external_id:
            self.template = self._get_one_template_or_nothing()
            if self.template:
                self.template_id = self.template.obj.id

        self.hydrated_event = self._get_hydrated_event()
        self.hydrated_template = self._get_hydrated_template()

    def get_name(self):
        if self.event:
            return self.event['name']
        if self.template:
            return self.template.obj.name
        return ''

    def get_duration(self):
        default = 60
        durations = [default]
        if self.event and self.event['startTs'] and self.event['endTs']:
            delta = parse(self.event['endTs']) - parse(self.event['startTs'])
            durations.append(round_duration(delta.total_seconds() / 60))
        if self.template:
            durations.append(self.hydrated_template['duration'])
        return max(durations)

    def get_record(self):
        return self.template.obj.record if self.template else False

    def get_stream(self):
        return self.template.obj.stream if self.template else False

    def get_unique_participants(self):
        participants_chain = []
        if self.hydrated_template:
            participants_chain.append(self.hydrated_template['participants'])
        if self.hydrated_event:
            participants_chain.append(self.hydrated_event['rooms'])
            if self.hydrated_event.get('zoom'):
                participants_chain.append([self.hydrated_event['zoom']])
        participants = chain(*participants_chain)

        seen = set()
        for participant in participants:
            key = (participant['id'], participant['type'])
            if key not in seen:
                seen.add(key)
                yield participant

    @property
    def data(self):
        return {
            'call': {
                'event_id': self.event_id,
                'template_id': self.template_id,
                'name': self.get_name(),
                'duration': self.get_duration(),
                'participants': self.get_unique_participants(),
                'record': self.get_record(),
                'stream': self.get_stream(),
            },
            'template': self.hydrated_template,
            'event': self.hydrated_event,
            'relevant_templates': [
                template.as_dict(hydrator=self.participants_hydrator)
                for template in self.relevant_templates
            ],
        }
