from uuid import uuid4

from datetime import timedelta
from django.db import models
from django.conf import settings
from django.contrib.postgres.fields import ArrayField
from model_utils import Choices

from django.utils import timezone

from intranet.vconf.src.call.constants import CALL_STATES, STREAM_LANGUAGES


class Event(models.Model):
    """модель предназначена для хранения мастер id и данных предстоящей встречи"""
    id = models.IntegerField(primary_key=True)
    external_id = models.CharField(max_length=256, null=True)
    name = models.CharField(max_length=250)
    description = models.TextField(default='', blank=True)
    start_time = models.DateTimeField()
    end_time = models.DateTimeField()
    secret = models.CharField(max_length=36, null=True)

    def generate_secret(self):
        self.secret = str(uuid4())


class CallBase(models.Model):
    class Meta:
        abstract = True
    name = models.CharField(max_length=250)
    chat = models.BooleanField(default=False)
    stream = models.BooleanField(default=False)
    record = models.BooleanField(default=False)
    duration = models.DurationField()
    # id встречи в Календаре
    event_id = models.IntegerField(null=True, db_index=True)
    # id серии встреч в Календаре
    event_external_id = models.CharField(max_length=256, null=True)
    next_event = models.ForeignKey(Event, on_delete=models.SET_NULL, null=True)
    priority = models.IntegerField(default=0, db_index=True)
    lang = models.CharField(max_length=3, choices=STREAM_LANGUAGES, default=STREAM_LANGUAGES.ru)


CALL_METHODS = Choices('cisco', 'email', 'mobile', 'messenger_q', 'zoom')
PARTICIPANT_TYPES = Choices('person', 'room', 'arbitrary', 'cms')


class ParticipantBase(models.Model):
    METHODS = CALL_METHODS + ['cms']
    TYPES = PARTICIPANT_TYPES

    class Meta:
        abstract = True
    obj_id = models.CharField(max_length=64, db_index=True)
    obj_type = models.CharField(max_length=64, choices=PARTICIPANT_TYPES)
    method = models.CharField(max_length=32, choices=METHODS, default=CALL_METHODS.cisco)


class Participant(ParticipantBase):
    STATES = Choices('active', 'disconnected', 'ended')
    conf_call = models.ForeignKey(
        'ConferenceCall',
        on_delete=models.CASCADE,
        related_name='participants',
    )
    cms_id = models.CharField(max_length=64, db_index=True)
    state = models.CharField(max_length=32, choices=STATES, default=STATES.disconnected)
    number = models.CharField(max_length=64)
    camera = models.BooleanField(default=True)
    microphone = models.BooleanField(default=True)
    camera_active = models.BooleanField(default=True)
    microphone_active = models.BooleanField(default=True)

    class Meta:
        ordering = ('id',)


class Record(models.Model):
    conf_call = models.ForeignKey(
        'ConferenceCall', on_delete=models.CASCADE, related_name='+'
    )
    node = models.CharField(max_length=64)
    file_name = models.CharField(max_length=64)
    is_deleted = models.BooleanField(default=False)


class CallTemplate(CallBase):
    master_template = models.ForeignKey(
        'CallTemplate',
        on_delete=models.CASCADE,
        blank=True,
        null=True,
        default=None,
    )
    owners = ArrayField(models.CharField(max_length=128))
    stream_picture = models.CharField(default='', blank=True, max_length=500)
    stream_description = models.TextField(default='', blank=True)


class ParticipantTemplate(ParticipantBase):
    call_template = models.ForeignKey(
        'CallTemplate', on_delete=models.CASCADE, related_name='+'
    )


class ConferenceCall(CallBase):
    class Meta:
        indexes = [
            models.Index(fields=['conf_cms_id', 'author_login', 'state']),
            models.Index(fields=['author_login', 'state']),
        ]
    STATES = CALL_STATES
    master_call = models.ForeignKey(
        'ConferenceCall',
        on_delete=models.CASCADE,
        blank=True,
        null=True,
        default=None,
    )
    conf_cms_id = models.CharField(max_length=64, primary_key=True)
    call_cms_id = models.CharField(max_length=64, default='')

    meeting_id = models.CharField(max_length=64)
    uri = models.CharField(max_length=32)
    secret = models.CharField(max_length=36)
    chat_invite_hash = models.CharField(max_length=36, null=True)

    start_time = models.DateTimeField(null=True, db_index=True)
    stop_time = models.DateTimeField(null=True)
    author_login = models.CharField(max_length=128, db_index=True)

    state = models.CharField(max_length=32, choices=STATES, default=STATES.broken, db_index=True)
    cms_node = models.CharField(max_length=128, default='')
    taken = models.BooleanField(default=False)

    template = models.ForeignKey(CallTemplate, null=True, blank=True, on_delete=models.SET_NULL)

    ether_back_id = models.CharField(max_length=64, default='')
    ether_front_id = models.CharField(max_length=64, default='')
    viewers_count = models.IntegerField(default=0)
    # приватная трансляция не отображается только в ручке calls/?with_stream
    is_private_stream = models.BooleanField(default=False)
    has_active_chat = models.BooleanField(default=False)

    @property
    def chat_id(self):
        return '0/{}/{}'.format(settings.MESSENGER_NAMESPACE_ID, self.uri)


class UserSettings(models.Model):

    login = models.CharField(max_length=128, unique=True)
    method = models.CharField(max_length=32, choices=CALL_METHODS)


class Node(models.Model):

    id = models.IntegerField(primary_key=True)
    vm_host = models.CharField(max_length=32, default='', blank=True)
    last_update = models.DateTimeField()
    load_value = models.IntegerField()
    load_limit = models.IntegerField()
    enabled = models.BooleanField(default=False)

    class Meta:
        ordering = ('id',)

    @property
    def api_url(self):
        return settings.CMS_API_URL_TEMPLATE.format(num=self.id)

    @property
    def load_available(self):
        return self.load_limit - self.load_value

    @property
    def is_alive(self):
        return self.last_update > timezone.now() - timedelta(minutes=2)

    def __str__(self):
        return f'Node {self.id}'
