import phonenumbers
import re

from django import forms
from django.conf import settings
from django.core.mail import send_mail

from intranet.vconf.src.call.models import Participant, CALL_METHODS
from intranet.vconf.src.call.manager import CallManager
from intranet.vconf.src.ext_api.api_hd import get_passcode_by_id

import logging
log = logging.getLogger(__name__)


class ParticipantCtl:
    def __init__(self, data):
        self._data = data

    @property
    def data(self) -> dict:
        return self._data

    def get_or_create_obj(self, call, state=Participant.STATES.disconnected) -> Participant:
        try:
            obj = (
                Participant.objects
                .get(
                    obj_id=self.data['id'],
                    obj_type=self.data['type'],
                    conf_call=call,
                )
            )
        except Participant.DoesNotExist:
            obj = Participant()
            obj.conf_call = call
            obj.obj_id = self.data['id']
            obj.obj_type = self.data['type']

        if 'cms_id' in self.data:
            obj.cms_id = self.data['cms_id']

        obj.number = self.data['number']
        obj.camera = self.data.get('camera', True)
        obj.microphone = self.data.get('microphone', True)
        obj.method = self.data['method']
        obj.state = state
        obj.save()

        self.obj = obj

        return obj

    def add_to_call(self, mngr: CallManager) -> None:
        self.get_or_create_obj(mngr.obj)

        if self.obj.method == Participant.METHODS.email:
            self._send_mail(mngr=mngr)
        elif self.obj.method == Participant.METHODS.cms:
            # нельзя позвонить в cms приложение
            return
        else:
            uri = self._get_uri()
            self._add_to_cms(mngr, uri)

        log.info(
            'The participant %s id==%s is added to the call %s',
            self.obj.obj_id, self.obj.cms_id, mngr.obj.conf_cms_id
        )
        self.obj.save()

    def _get_uri(self):
        raise RuntimeError

    def _get_email(self):
        raise RuntimeError

    def _send_mail(self, mngr: CallManager) -> None:
        send_mail(
            'Meeting invitation',
            settings.EMAIL_TEXT.format(link=mngr.invite_link),
            'vconf@yandex-team.ru',
            [self._get_email()],
            fail_silently=False,
        )
        self.obj.state = Participant.STATES.disconnected

    def _add_to_cms(self, mngr: CallManager, uri: str) -> None:
        label = self.data.get('label', '')
        res = mngr.cms.participant_add(uri=uri, call_id=mngr.obj.call_cms_id, label=label)
        if res.status_code != 200:
            raise mngr.Error('bad request ' + str(res.status_code) + res.text)
        self.obj.cms_id = res.headers['Location'][21:]
        self.obj.state = Participant.STATES.active


class PersonParticipant(ParticipantCtl):

    @staticmethod
    def get_actions(ext_data: dict) -> list:
        actions = []
        if ext_data['work_phone']:
            actions.append(CALL_METHODS.cisco)
        for phone in ext_data['phones']:
            if phone['kind'] == 'common' and phone['protocol'] != 'sms':
                actions.append(CALL_METHODS.mobile)
                break
        actions.append(CALL_METHODS.email)
        actions.append(CALL_METHODS.messenger_q)
        return actions

    @classmethod
    def _cut_label(cls, long_label):
        label = ''
        count = 0
        for s in long_label:
            count += len(s.encode('utf8'))
            if count > 50:
                break
            label += s
        return label

    @classmethod
    def build_label(cls, data):
        first_name = data['name']['first'].get('en', '')
        second_name = data['name']['last'].get('en', '')

        if not (first_name or second_name):
            label = data['login']
        else:
            label = f'{first_name} {second_name}'

        if len(label.encode('utf8')) > 50:
            label = cls._cut_label(label)

        label = label.strip()
        return label

    @classmethod
    def from_raw(cls, raw_data: dict, ext_data: dict, lang: str):
        if not ext_data:
            log.error('ext_data for `%s` is empty', raw_data['id'])
            ext_data = {}

        raw_data.update({
            'name': '{} {}'.format(
                ext_data['name']['first'].get(lang, ''),
                ext_data['name']['last'].get(lang, '')
            ),
            'label': cls.build_label(ext_data),
            'number': str(raw_data.get(
                'number',
                ext_data['work_phone'] if ext_data['work_phone'] else ''
            )),
            'camera': raw_data.get('camera', True),
            'microphone': raw_data.get('microphone', True),
            'action': cls.get_actions(ext_data),
            'method': raw_data.get('method', None),
            'tz': ext_data['environment']['timezone'],
            'settings': ext_data.get('settings', {}),
        })
        return cls(data=raw_data)

    def _get_email(self) -> str:
        return '%s@yandex-team.ru' % self.obj.obj_id

    def _get_uri(self) -> str:
        if self.obj.method == Participant.METHODS.messenger_q:
            uri = self.obj.obj_id + '@q.yandex-team.ru'
        else:
            uri = self.obj.number + '@yandex-team.ru'

        if self.obj.method == Participant.METHODS.mobile:
            uri = '55' + uri
        return uri


class RoomParticipant(ParticipantCtl):
    @classmethod
    def from_raw(cls, raw_data: dict, ext_data: dict, lang: str):
        _id = str(raw_data['id'])
        video_number = ext_data.get('equipment', {}).get('video_conferencing', '')

        raw_data.update({
            'id': _id,
            'name': ext_data.get('name', {}).get(lang, raw_data.get('number', '')),
            'label': ext_data.get('name', {}).get('en', raw_data.get('number', '')),
            'number': str(raw_data.get(
                'number',
                video_number or ext_data.get('phone', '')
            )),
            'camera': raw_data.get('camera', bool(video_number)),
            'microphone': raw_data.get('microphone', True),
            'action': [CALL_METHODS.cisco],
            'method': raw_data.get('method', None),
        })
        return cls(data=raw_data)

    def _get_uri(self) -> str:
        return self.obj.number + '@yandex-team.ru'


class ParticipantValidationError(Exception):
    pass


class ArbitraryParticipant(ParticipantCtl):

    @staticmethod
    def _normalize_phone(phone):
        # international number
        if phone.startswith('9810'):
            phone = '+' + phone[4:]
        # russian number
        elif phone.startswith('98'):
            phone = '+7' + phone[2:]

        try:
            parsed = phonenumbers.parse(phone, region='RU')
        except phonenumbers.NumberParseException:
            return

        if not phonenumbers.is_valid_number(parsed):
            log.warning('Number %s is not valid', phone)
            return phone

        if not phonenumbers.is_possible_number(parsed):
            log.warning('Number %s is not possible', phone)
            return phone

        return phonenumbers.format_number(
            numobj=parsed,
            num_format=phonenumbers.PhoneNumberFormat.E164,
        )

    @staticmethod
    def _normalize_email(email):
        try:
            return forms.EmailField().clean(email).lower()
        except forms.ValidationError:
            return

    @staticmethod
    def _parse_id(cid: str) -> str:
        try:
            parsed_id = re.search(r'(?P<mid>\d{9,11})', cid)
            if parsed_id:
                logging.debug('Parsed  %s', parsed_id.groupdict())
                return parsed_id.groupdict().get('mid')
        except Exception as e:
            logging.exception('%s during %s parsing', e, cid)

    @staticmethod
    def normalize_zoom_url(url):
        url = url.lower()
        if url.startswith(settings.ZOOM_URL):
            id_string = ArbitraryParticipant._parse_id(url[len(settings.ZOOM_URL):])
            try:
                zoom_id = int(id_string)
            except TypeError:
                return

            sip_pwd = get_passcode_by_id(id_string)

            return f'{zoom_id}.{sip_pwd}@zoom' if sip_pwd else f'{zoom_id}@zoom'

    @classmethod
    def _get_normalized_id_and_actions(cls, _id: str):
        _id = _id.strip()

        normalized_id = cls.normalize_zoom_url(_id)
        if normalized_id:
            return normalized_id, [CALL_METHODS.zoom]

        normalized_id = cls._normalize_phone(_id)
        if normalized_id:
            return normalized_id, [CALL_METHODS.mobile]

        # Чтобы иметь возможность позвонить на абсолютно любой SIP-URI,
        # а не только на тот, который парсится как email.
        if _id.startswith('sip:') or _id.startswith('sips:'):
            return _id, [CALL_METHODS.mobile]

        if _id.endswith('@zoom'):
            return _id, [CALL_METHODS.zoom]

        normalized_id = cls._normalize_email(_id)
        if normalized_id:
            if any(domain in normalized_id for domain in ('@zoomcrc.com', '@zmeu.us')):
                actions = [CALL_METHODS.zoom]
            else:
                actions = [CALL_METHODS.email, CALL_METHODS.mobile]
            return normalized_id, actions
        else:
            raise ParticipantValidationError(f'Invalid arbitrary id {_id}')

    @classmethod
    def from_raw(cls, raw_data: dict, safe=False):
        try:
            normalized_id, actions = cls._get_normalized_id_and_actions(raw_data['id'])
        except ParticipantValidationError:
            if safe:
                normalized_id, actions = raw_data['id'], []
            else:
                raise

        raw_data.update({
            'id': normalized_id,
            'name': normalized_id,
            'number': normalized_id,
            'label': normalized_id,
            'camera': False,
            'microphone': True,
            'action': actions,
            'method': raw_data.get('method', None),
        })
        return cls(data=raw_data)

    def _get_uri(self) -> str:
        return self.obj.obj_id

    def _get_email(self) -> str:
        return self.obj.obj_id


class CmsParticipant(ParticipantCtl):
    @classmethod
    def from_raw(cls, raw_data):
        raw_data['name'] = raw_data['id']
        raw_data['action'] = ['cms']
        return cls(data=raw_data)
