import json
import logging
from functools import partial

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db.models import ObjectDoesNotExist, Prefetch
from django.http import HttpResponse

from intranet.vconf.src.call.constants import CallFilter
from intranet.vconf.src.call.models import ConferenceCall, Participant, PARTICIPANT_TYPES
from intranet.vconf.src.call.manager import CallManager
from intranet.vconf.src.lib.json import responding_json, make_json_response
from intranet.vconf.src.lib.exceptions import HttpError
from intranet.vconf.src.lib.utils import hydrate_participants, round_duration
from intranet.vconf.src.ext_api.staff import get_from_staff_api


log = logging.getLogger(__name__)
User = get_user_model()
_make_json_response = partial(make_json_response, data_field_name='data')


@responding_json
def add_record(request):
    try:
        req_data = json.loads(request.read().decode('utf-8'))
    except ValueError as e:
        log.exception('save_record bad request')
        return str(e), 400
    else:
        log.debug('Request data: %s', req_data)

    try:
        call = CallManager.get_by_id(req_data['space_id'])
    except ObjectDoesNotExist:
        log.exception('Call with space_id=%s not found', req_data['space_id'])
        return 'Call not found', 404

    call.add_record(data=req_data)

    return 'Record was saved successfully'


def old_create(request):
    event_id = request.GET.get('event_id')
    if event_id:
        try:
            event_id = int(event_id)
        except ValueError:
            return _make_json_response(data=[{'error': 'event_id must be an integer'}], status=400)
        call = (
            ConferenceCall.objects
            .filter(state=ConferenceCall.STATES.active, event_id=event_id)
            .first()
        )
        if call:
            return _make_json_response(data={
                'id': call.name,
                'cms_id': call.conf_cms_id,
            })

    errors = []

    author_login = request.GET.get('author_login')
    if not author_login:
        errors.append({'error': 'Author login is required'})

    phones = [p.strip() for p in request.GET.get('phones', '').split(',') if p]
    if not phones:
        errors.append({'error': 'Phones is required'})

    if errors:
        return _make_json_response(data=errors, status=400)

    participants = _get_participants(phones=phones)

    if not participants:
        return _make_json_response(data=[{'error': 'Rooms not found'}], status=404)

    participants = hydrate_participants(participants)

    duration = round_duration(request.GET.get('duration', 720))

    conf_data = {
        'duration': duration,
        'stream': False,
        'record': False,
        'participants': participants,
        'event_id': event_id,
    }

    if request.GET.get('fake'):
        return _make_json_response(data=conf_data)

    try:
        author, _ = User.objects.get_or_create(username=author_login)
        call = CallManager.create(data=conf_data, author=author)
    except CallManager.Error as e:
        return _make_json_response(data=[{'error': str(e)}], status=500)
    else:
        return _make_json_response(data={
            'id': call.obj.name,
            'cms_id': call.obj.conf_cms_id,
        })


def _get_participants(phones):
    phones_string = ','.join('"%s"' % phone for phone in phones)

    rooms = get_from_staff_api(
        path='/rooms/',
        params={
            '_fields': ','.join(['id', 'phone', 'equipment.video_conferencing']),
            '_query': 'phone in [{phones}] or equipment.video_conferencing in [{phones}]'.format(phones=phones_string)
        }
    )
    rooms_map = {}
    for room in rooms['result']:
        if room['phone']:
            rooms_map[room['phone']] = room['id']
        if room['equipment']['video_conferencing']:
            rooms_map[room['equipment']['video_conferencing']] = room['id']

    participants = [{'type': 'room', 'id': rooms_map[phone], 'method': 'cisco'} for phone in phones if phone in rooms_map]
    return participants


@responding_json
def download_record(request, conf_cms_id):
    call = CallManager.find_ended(
        for_user=request.user,
        conf_cms_id=conf_cms_id,
    )
    if call:
        try:
            record = call[0].get_record()
        except CallManager.PermissionDenied:
            raise HttpError(403)
    if not call or not record:
        raise HttpError(404, message='Record not found')

    response = HttpResponse(status=380)
    file_name = record.file_name.replace('\n', '').replace('\r', '')
    response['X-Record-Url'] = f'{record.node}/recs/spaces/{conf_cms_id}/{file_name}'
    response['X-Record-Filename'] = file_name
    return response


@responding_json
def stream_list(request):
    request_data = request.GET

    filter_data = {
        'show_all': True,
        'state': request_data.get('state'),
        'limit': int(request_data.get('limit', 20)),
        'page': int(request_data.get('page', 1)),
        'with_stream': True,
    }
    if filter_data['state'] == ConferenceCall.STATES.active:
        filter_data['with_active_participants'] = True

    calls = CallManager.find_calls(
        for_user=request.user,
        call_filter=CallFilter(**filter_data),
        base_qs=(
            ConferenceCall.objects
            .prefetch_related(
                Prefetch(
                    lookup='participants',
                    queryset=Participant.objects.filter(obj_type=PARTICIPANT_TYPES.room),
                    to_attr='rooms',
                ),
            )
            .order_by('-start_time')
        )
    )

    data = []
    for call in calls:
        data.append({
            'id': call.obj.conf_cms_id,
            'rtmp_url': settings.CMS_CONFIG['streaming_url'] + call.obj.uri,
            'name': call.obj.name,
            'state': call.obj.state,
            'ether_back_id': call.obj.ether_back_id,
            'ether_front_id': call.obj.ether_front_id,
            'rooms': [
                {'id': room.obj_id}
                for room in call.obj.rooms
            ],
        })
    return data


@responding_json
def active_call_list(request):
    """
    Ручка для планшетов у переговорок, чтобы понимать,
    создали для встречи звонок или нет (VCONF-1038)
    """
    return CallManager.find_active_with_participant_count()
