import json
import logging
import yenv

from django import forms
from django.db.models import ObjectDoesNotExist
from django.views.decorators.http import require_http_methods

from intranet.vconf.src.call.constants import CALL_STATES, CallFilter
from intranet.vconf.src.call.hydrator import ParticipantsHydrator
from intranet.vconf.src.call.participant_ctl import ParticipantValidationError
from intranet.vconf.src.call.manager import CallManager
from intranet.vconf.src.call.merger import CallDataMerger
from intranet.vconf.src.call.models import ConferenceCall, UserSettings, Record
from intranet.vconf.src.call.user import update_user_settings
from intranet.vconf.src.call.ether import AllStreamersAreBusy
from intranet.vconf.src.lib.exceptions import HttpError
from intranet.vconf.src.lib.json import responding_json
from intranet.vconf.src.ext_api.calendar import (
    CalendarError,
    EventDoesNotExist,
    EventIsNotAvailable,
)


log = logging.getLogger(__name__)


@responding_json
def current_user(request):
    participant = {
        'type': 'person',
        'id': request.user.login,
        'uid': request.user.uid,
        'lang': request.user.lang,
        'is_admin': request.user.is_admin,
        'affiliation': request.user.affiliation,
        'is_external': request.user.is_external,
        'is_ip_external': request.user.is_ip_external,
        'environment': yenv.type,
        'settings': {},
    }

    user_settings = UserSettings.objects.filter(login=request.user.login).first()
    if user_settings:
        participant['settings']['method'] = user_settings.method

    hydrator = ParticipantsHydrator(lang=request.user.lang)
    hydrator.add_to_fetch([participant])
    participant = hydrator.hydrate(participant)
    return participant.data


@responding_json
@require_http_methods(['POST'])
def user_settings(request):
    try:
        req_data = json.loads(request.read().decode('utf-8'))
    except ValueError as e:
        log.exception('user_settings bad request')
        raise HttpError(400, message=str(e))
    else:
        log.debug('Request data: %s', req_data)

    user_settings = update_user_settings(
        login=request.user.login,
        method=req_data['method'],
    )
    return {
        'method': user_settings.method,
    }


@responding_json
def call_create_meta(request):
    req_data = request.GET
    event_id = None
    template_id = None
    try:
        if 'event_id' in req_data:
            event_id = int(req_data['event_id'])
        if 'template_id' in req_data:
            template_id = int(req_data['template_id'])
    except ValueError:
        raise HttpError(400, code='validation_error')

    call_data_merger = CallDataMerger(
        user=request.user,
        event_id=event_id,
        template_id=template_id,
    )
    try:
        data = call_data_merger.data
    except EventDoesNotExist:
        raise HttpError(404, code='event_does_not_exist')
    except EventIsNotAvailable:
        raise HttpError(403, code='event_is_not_available')
    except CalendarError:
        raise HttpError(400, code='calendar_error')
    except ObjectDoesNotExist:
        raise HttpError(404, message='Template not found')
    return data


@responding_json
def call_create(request):
    try:
        req_data = json.loads(request.read().decode('utf-8'))
    except ValueError as e:
        log.exception('call_create bad request')
        raise HttpError(400, message=str(e))
    else:
        log.debug('Request data: %s', req_data)

    hydrator = ParticipantsHydrator(lang=request.user.lang)
    hydrator.add_to_fetch(req_data['participants'])

    try:
        req_data['participants'] = [
            hydrator.hydrate(pt)
            for pt in req_data['participants']
        ]
    except ParticipantValidationError:
        log.exception('Invalid participants')
        return 'Invalid participants', 400

    if not req_data['participants']:
        return 'Bad request participants is empty', 400

    try:
        call = CallManager.create(author=request.user, data=req_data)
    except CallManager.CallAlreadyExists as e:
        raise HttpError(400, message=str(e), code='call_already_exists')
    except CallManager.Error as e:
        raise HttpError(500, message=str(e), code='unexpected_error')
    except AllStreamersAreBusy:
        raise HttpError(400, message='All stream channels are busy', code='no_available_channels')

    return call.as_dict(hydrator=hydrator)


@responding_json
def meta(request):
    try:
        req_data = json.loads(request.read().decode('utf-8'))
    except ValueError as e:
        log.exception('call_create bad request')
        raise HttpError(400, message=str(e))
    else:
        log.debug('Request data: %s', req_data)

    hydrator = ParticipantsHydrator(lang=request.user.lang)
    hydrator.add_to_fetch(req_data['participants'])

    errors = []
    participants = []

    for pt in req_data['participants']:
        try:
            participants.append(hydrator.hydrate(pt).data)
        except ParticipantValidationError as e:
            errors.append({
                'participant': pt,
                'error': str(e),
                'code': 400,
            })

    if errors:
        return errors, 400
    else:
        return participants


def clean_call_list_params(request_data: dict) -> CallFilter:
    boolean_field = forms.BooleanField(required=False)
    call_filter = CallFilter(
        show_all=boolean_field.clean(request_data.get('show_all')),
        state=(
            CALL_STATES.ended
            if boolean_field.clean(request_data.get('history'))
            else request_data.get('state', CALL_STATES.active)
        ),
        limit=int(request_data.get('limit', 20)),
        page=int(request_data.get('page', 1)),
        with_record=boolean_field.clean(request_data.get('with_record')) if 'with_record' in request_data else None,
        with_stream=boolean_field.clean(request_data.get('with_stream')) if 'with_stream' in request_data else None,
        sort=request_data['sort'] if request_data.get('sort') in {'priority', '-priority'} else None,
    )
    return call_filter


@responding_json
def call_list(request):
    hydrator = ParticipantsHydrator(lang=request.user.lang)
    call_filter = clean_call_list_params(request.GET)
    base_qs = ConferenceCall.objects.prefetch_related('participants')

    if call_filter.with_stream:
        base_qs = base_qs.exclude(is_private_stream=True)

    calls = CallManager.find_calls(
        for_user=request.user,
        base_qs=base_qs,
        call_filter=call_filter,
    )
    return [call.as_dict(hydrator=hydrator) for call in calls]


@responding_json
def call_detail(request, conf_cms_id):
    calls = CallManager.find_active(
        for_user=request.user,
        conf_cms_id=conf_cms_id,
    )
    if calls:
        return calls[0].as_detail_dict()
    else:
        return 'Call id=%s not found' % conf_cms_id, 404


@responding_json
def add_participant(request, conf_cms_id):
    try:
        req_data = json.loads(request.read().decode('utf-8'))
    except ValueError as e:
        log.exception('add_participant bad request')
        raise HttpError(400, message=str(e))
    else:
        log.debug('Request data: %s', req_data)

    hydrator = ParticipantsHydrator(lang=request.user.lang)
    hydrator.add_to_fetch(req_data['participants'])

    errors = []
    participants = []
    for pt in req_data['participants']:
        try:
            participants.append(hydrator.hydrate(pt))
        except ParticipantValidationError as e:
            errors.append({
                'participant': pt,
                'error': str(e),
                'code': 400,
            })

    if errors:
        return errors, 400

    calls = CallManager.find_active(
        for_user=request.user,
        conf_cms_id=conf_cms_id,
    )
    if calls:
        call = calls[0]
        broken = []
        for participant in participants:
            try:
                call.add_participant(participant)
            except CallManager.PermissionDenied as e:
                return str(e), 403
            except CallManager.Error as e:
                broken.append({
                    'participant': participant.data,
                    'error': str(e),
                    'code': 500,
                })
        if broken:
            return broken
        else:
            return call.as_dict()
    else:
        return 'Call with id=%s not found' % conf_cms_id, 404


@responding_json
def toggle_microphone(request, conf_cms_id):
    try:
        req_data = json.loads(request.read().decode('utf-8'))
    except ValueError as e:
        log.exception('toggle_microphone bad request')
        raise HttpError(400, message=str(e))
    else:
        log.debug('Request data: %s', req_data)

    calls = CallManager.find_active(
        for_user=request.user,
        conf_cms_id=conf_cms_id,
    )
    if calls:
        call = calls[0]
        try:
            call.toggle_microphone(req_data)
        except CallManager.PermissionDenied as e:
            return str(e), 403
        except CallManager.Error as e:
            return str(e), 500
        else:
            return call.as_dict()
    else:
        return 'Call with id=%s not found' % conf_cms_id, 404


@responding_json
def toggle_camera(request, conf_cms_id):
    try:
        req_data = json.loads(request.read().decode('utf-8'))
    except ValueError as e:
        log.exception('toggle_camera bad request')
        raise HttpError(400, message=str(e))
    else:
        log.debug('Request data: %s', req_data)

    calls = CallManager.find_active(
        for_user=request.user,
        conf_cms_id=conf_cms_id,
    )
    if calls:
        call = calls[0]
        try:
            call.toggle_camera(req_data)
        except CallManager.PermissionDenied as e:
            return str(e), 403
        except CallManager.Error as e:
            return str(e), 500
        else:
            return call.as_dict()
    else:
        return 'Call with id=%s not found' % conf_cms_id, 404


def remove_or_disconnect_participant(request, conf_cms_id, action):
    try:
        req_data = json.loads(request.read().decode('utf-8'))
    except ValueError as e:
        log.exception('%s participant bad request', action)
        raise HttpError(400, message=str(e))
    else:
        log.debug('Request data: %s', req_data)

    calls = CallManager.find_active(
        for_user=request.user,
        conf_cms_id=conf_cms_id,
    )
    if calls:
        call = calls[0]
        only_disconnect = True if action != 'remove' else False
        try:
            call.remove_participant(req_data, only_disconnect=only_disconnect)
        except CallManager.PermissionDenied as e:
            return str(e), 403
        except CallManager.ParticipantNotFound as e:
            raise HttpError(
                status=400,
                code='participant_not_found',
                message=f'can not {action} participant {str(e)}'
            )
        except CallManager.Error as e:
            log.exception('Failed to %s participant', action)
            return str(e), 500
        else:
            return call.as_dict()
    else:
        return 'Call with id=%s not found' % conf_cms_id, 404


@responding_json
def remove_participant(request, conf_cms_id):
    return remove_or_disconnect_participant(request, conf_cms_id, action='remove')


@responding_json
def disconnect_participant(request, conf_cms_id):
    return remove_or_disconnect_participant(request, conf_cms_id, action='disconnect')


@responding_json
def call_duration(request, conf_cms_id):
    try:
        req_data = json.loads(request.read().decode('utf-8'))
        duration = req_data['duration']
    except (ValueError, KeyError) as e:
        log.exception('Duration bad request')
        raise HttpError(400, message=str(e))
    else:
        log.debug('Request data: %s', req_data)

    calls = CallManager.find_active(
        for_user=request.user,
        conf_cms_id=conf_cms_id,
    )

    if calls:
        call = calls[0]
        try:
            call.update_duration(duration)
        except CallManager.PermissionDenied as e:
            return str(e), 403
        except CallManager.Error as e:
            return str(e), 500
        else:
            return call.as_dict()
    else:
        return 'Active call with id=%s not found' % conf_cms_id, 404


@responding_json
def call_stop(request, conf_cms_id):
    calls = CallManager.find_active(
        for_user=request.user,
        conf_cms_id=conf_cms_id,
    )
    if calls:
        call = calls[0]
        try:
            call.stop()
        except CallManager.PermissionDenied as e:
            return str(e), 403
        except CallManager.Error as e:
            return str(e), 500
        else:
            return call.as_dict()
    else:
        return 'Active call with id=%s not found' % conf_cms_id, 404


@responding_json
@require_http_methods(['DELETE'])
def delete_record(request, conf_cms_id):
    try:
        call = CallManager.get_by_id(conf_cms_id=conf_cms_id, for_user=request.user)
    except ConferenceCall.DoesNotExist:
        raise HttpError(404, code='call_does_not_exist')

    try:
        call.delete_record()
    except Record.DoesNotExist:
        raise HttpError(404, code='record_does_not_exist')
    except CallManager.PermissionDenied:
        raise HttpError(403)

    return 'Record was successfully deleted', 200
