import logging

from django import forms
from django.contrib.auth import get_user_model
from django.views.decorators.http import require_http_methods

from intranet.vconf.src.lib.json import responding_json
from intranet.vconf.src.call.constants import CallFilter, CALL_STATES
from intranet.vconf.src.call.hydrator import EventsHydrator
from intranet.vconf.src.call.manager import CallManager
from intranet.vconf.src.call.event import user_is_event_participant, update_or_create_event
from intranet.vconf.src.ext_api.calendar import (
    get_user_events,
    get_event_info,
    get_next_event,
    CalendarError,
    EventDoesNotExist,
    EventIsNotAvailable,
)
from intranet.vconf.src.call.models import Event
from intranet.vconf.src.lib.exceptions import HttpError


log = logging.getLogger(__name__)
User = get_user_model()


@responding_json
def event_list(request):
    events = get_user_events(user=request.user)['events']
    hydrator = EventsHydrator(user=request.user)
    hydrator.add_to_fetch(events)
    return [hydrator.hydrate(event).as_dict() for event in events]


@responding_json
def event_detail(request, event_id):
    try:
        event = get_event_info(event_id=event_id, user=request.user)
    except EventDoesNotExist as e:
        return str(e), 404
    except EventIsNotAvailable as e:
        return str(e), 403
    except CalendarError as e:
        return str(e), 500

    hydrator = EventsHydrator(user=request.user)
    hydrator.add_to_fetch([event])
    return hydrator.hydrate(event).as_dict()


@responding_json
@require_http_methods(['POST'])
def event_generate_secret(request, event_id):
    request_data = request.GET
    boolean_field = forms.BooleanField(required=False)
    force = boolean_field.clean(request_data.get('force', False))

    try:
        event_data = get_next_event(event_id=event_id, user=request.user)
        if not (request.user.is_admin or user_is_event_participant(request.user, event_data)):
            raise HttpError(403)
    except CalendarError:
        raise HttpError(400, code='calendar_error', message='Calendar API error')

    event = update_or_create_event(event_data, regenerate_secret=force)
    return {
        'event_id': event_id,
        'master_id': event.id,
        'secret': event.secret,
    }


def _check_event(master_id, secret):
    try:
        event = Event.objects.get(id=master_id)
    except Event.DoesNotExist:
        raise HttpError(404, code='event_does_not_exist')

    if event.secret != secret:
        raise HttpError(403, code='wrong_secret')


def _handle_no_active_call(master_id, is_authenticated):
    details = {
        'is_authenticated': is_authenticated,
    }
    try:
        event_data = get_event_info(event_id=master_id)
        details['event'] = {
            'startTs': event_data['startTs'],
            'id': event_data['id'],
            'master_id': master_id,
        }
        if is_authenticated:
            details['event']['name'] = event_data['name']
    except CalendarError:
        log.error(f'Calendar error for event `{master_id}`')
    raise HttpError(
        status=404,
        code='no_active_call_for_event_master_id',
        details=details,
    )


@responding_json
@require_http_methods(['GET'])
def event_invite(request, event_id):
    secret = request.GET.get('secret')
    master_id = request.GET.get('master_id')
    if master_id:
        master_id = int(master_id)

    _check_event(master_id, secret)
    is_authenticated = request.yauser.is_authenticated()
    user = request.user if is_authenticated else User(extra={'secret': secret})

    calls = CallManager.find_calls(
        for_user=user,
        call_filter=CallFilter(
            state=CALL_STATES.active,
            next_event_id=master_id,
        )
    )
    if calls:
        call = calls[0]
        return {
            'invite_link': call.invite_link,
            'is_authenticated': is_authenticated,
        }
    else:
        _handle_no_active_call(master_id, is_authenticated)
