import json
import logging

from django.views.decorators.http import require_http_methods
from django.forms import ValidationError

from intranet.vconf.src.call.call_template import Template
from intranet.vconf.src.call.constants import CallFilter, CALL_STATES
from intranet.vconf.src.call.hydrator import ParticipantsHydrator
from intranet.vconf.src.call.manager import CallManager
from intranet.vconf.src.call.models import CallTemplate
from intranet.vconf.src.lib.exceptions import HttpError
from intranet.vconf.src.lib.json import responding_json
from intranet.vconf.src.ext_api.calendar import (
    parse_event_id,
    get_next_event,
    CalendarError,
)


log = logging.getLogger(__name__)


def validate_template_data(data):
    if data.get('stream_picture'):
        if not data['stream_picture'].startswith('/jing/files/'):
            raise ValidationError('invalid stream picture')
        if len(data['stream_picture']) > 500:
            raise ValidationError('invalid link length')


@responding_json
def save_template(request, obj_id=None):
    if request.method == 'GET':
        if obj_id is None:
            raise HttpError(400, message='bad request obj_id must be not None')
        items = Template.find(for_user=request.user, obj_id=obj_id)
        if items:
            return items[0].as_detail_dict(user=request.user)
        else:
            raise HttpError(404, message='Template not found')

    try:
        req_data = json.loads(request.read().decode('utf-8'))
    except ValueError as e:
        log.exception('save_template bad request')
        raise HttpError(400, message=str(e))
    else:
        log.debug('Request data: %s', req_data)

    try:
        validate_template_data(req_data)
    except ValidationError as e:
        raise HttpError(400, code='validation_error', message=str(e))

    if not req_data['owners']:
        req_data['owners'].append(request.user.login)

    if req_data.get('event_id'):
        event_id = parse_event_id(req_data['event_id'])
        if not event_id:
            raise HttpError(400, message='Calendar parse event_id error')
        try:
            event = get_next_event(event_id=event_id, user=request.user)
        except CalendarError:
            raise HttpError(400, code='calendar_error', message='Calendar API error')

        req_data['event'] = event
        req_data['event_id'] = event_id
        req_data['event_external_id'] = event['externalId']
    else:
        # Если event_id не передан, то event_external_id надо тоже обнулить
        req_data['event_external_id'] = None
        req_data['event'] = {}

    if obj_id is None:
        item = Template.create(req_data)
    else:
        items = Template.find(for_user=request.user, obj_id=obj_id)
        if items:
            item = items[0]
            item.update(req_data)
        else:
            raise HttpError(404, message='Template not found')

    return item.as_dict(lang=request.user.lang)


def _template_list(lang, user):
    hydrator = ParticipantsHydrator(lang=lang)
    items = Template.find(for_user=user)
    return [item.as_dict(hydrator=hydrator) for item in items]


@responding_json
def template_list(request):
    return _template_list(lang=request.user.lang, user=request.user)


@responding_json
def delete_template(request, obj_id):
    items = Template.find(for_user=request.user, obj_id=obj_id)
    if items:
        item = items[0]
        item.delete()
        return _template_list(lang=request.user.lang, user=request.user)
    else:
        return 'Template not found', 404


def _handle_active_call(call):
    if not call.obj.stream:
        data = {
            'code': 'call_is_not_stream',
            'message': 'Call is not a stream',
        }
        return data, 404
    else:
        return call.as_detail_dict()


@responding_json
@require_http_methods(['GET'])
def templates_with_stream(request):
    request_data = request.GET
    items = Template.find_streams(
        limit=int(request_data.get('limit', 20)),
        page=int(request_data.get('page', 1)),
    )
    return [item.as_brief_dict(tz=request.user.tz) for item in items]


@responding_json
@require_http_methods(['GET'])
def template_stream(request, obj_id):
    # Тут приходится делать запрос в обход менеджера Template,
    # потому что нам нужен event_external_id вне зависимости от доступов к шаблону
    try:
        template = CallTemplate.objects.get(id=obj_id)
    except CallTemplate.DoesNotExist:
        raise HttpError(
            status=404,
            code='template_is_not_found',
            message='Template is not found',
        )

    event_id = template.event_id
    event_external_id = template.event_external_id

    if not event_external_id:
        calls = CallManager.find_calls(
            for_user=request.user,
            call_filter=CallFilter(
                state=CALL_STATES.active,
                template_id=obj_id,
            )
        )
        if calls:
            return _handle_active_call(calls[0])
        else:
            raise HttpError(
                status=404,
                code='no_active_calls_for_template',
                message='There are no active calls for this template'
            )
    else:
        calls = CallManager.find_calls(
            for_user=request.user,
            call_filter=CallFilter(
                state=CALL_STATES.active,
                event_external_id=event_external_id,
                show_all=True,
                with_stream=True,
            )
        )
        if calls:
            return _handle_active_call(calls[0])
        else:
            try:
                event = get_next_event(event_id=event_id, user=request.user)
                details = {
                    'event': {
                        'startTs': event['startTs'],
                        'id': event['master_id'],
                        'name': event['name']
                    }
                }
            except CalendarError:
                details = {}
            raise HttpError(
                status=404,
                code='no_active_calls_for_event_external_id',
                message='There are not active calls for this event_external_id',
                details=details,
            )
