import asyncio
import sqlalchemy as sa

from intranet.vconf.src.cms_sync.log import logger
from intranet.vconf.src.cms_sync.settings import CMS_URL_TEMPLATE


calls = sa.Table(
    'call_conferencecall', sa.MetaData(),
    sa.Column('call_cms_id', sa.String(64)),
    sa.Column('state', sa.String(32)),
    sa.Column('cms_node', sa.String(32)),
)


participants = sa.Table(
    'call_participant', sa.MetaData(),
    sa.Column('cms_id', sa.String(64)),
    sa.Column('state', sa.String(32)),
    sa.Column('microphone_active', sa.Boolean, default=True),
    sa.Column('camera_active', sa.Boolean, default=True),
)


class Gateway:

    def __init__(self, conn):
        self.conn = conn

    async def end_call(self, call_id):
        query = (
            calls
            .update()
            .where(calls.c.call_cms_id == call_id)
            .values(state='ended_in_cms')
            .returning(calls.c.call_cms_id)
        )
        result = await self.conn.scalar(query)

        if result:
            logger.info('[DB] Call `%s` marked as ended.', call_id)
        else:
            logger.info('[DB] Call `%s` does not exist in db.', call_id)

        return result

    async def disconnect_participant(self, participant_id):
        """
        Дисконнектим участника, если он уже не удален из конференции.

        В том случае, когда событие апдейта прилетает после действия через интерфейс
        (который переводит участника в статус `ended`), теоретически возможно 2 случая:

        1. Мы пытаемся выполнить этот запрос до того, как произошел апдейт в конкурирующем запросе.
           Тогда этот запрос выполнится, а параллельный запрос перезапишет на ended. Что ОК.

        2. Мы пытаемся выполнить этот запрос после апдейта в параллельном запросе, но до коммита транзакции.
           Тогда мы либо получим lock timeout и это ОК, потому что ничего уже делать не нужно.
           Либо дождемся завершения транзакции, но участник будет уже `ended` и мы апдейтить не будем.
        """
        query = (
            participants
            .update()
            .where(
                sa.and_(
                    participants.c.cms_id == participant_id,
                    participants.c.state != 'ended',
                )
            )
            .values(state='disconnected')
            .returning(participants.c.cms_id)
        )
        await asyncio.sleep(0.5)
        result = await self.conn.scalar(query)

        if result:
            logger.info('[DB] Participant `%s` was disconnected.', participant_id)
        else:
            logger.info('[DB] Participant `%s` does not exist in db or is already ended.', participant_id)

        return result

    async def _update_audio_or_video(self, participant_id, field_name, value):
        query = (
            participants
            .update()
            .where(participants.c.cms_id == participant_id)
            .values({field_name: value})
            .returning(participants.c.cms_id)
        )

        result = await self.conn.scalar(query)

        if result:
            logger.info('[DB] %s set to %s for participant `%s`', field_name, value, participant_id)
        else:
            logger.info('[DB] Participant `%s` does not exist in db.', participant_id)

        return result

    async def update_participant(self, participant_id, data):
        if 'audioMuted' in data:
            await self._update_audio_or_video(
                participant_id=participant_id,
                field_name='microphone_active',
                value=not data['audioMuted'],
            )
        if 'videoMuted' in data:
            await self._update_audio_or_video(
                participant_id=participant_id,
                field_name='camera_active',
                value=not data['videoMuted'],
            )

    async def get_active_calls(self, node):
        query = (
            calls
            .select()
            .where(
                sa.and_(
                    calls.c.cms_node == CMS_URL_TEMPLATE.format(node=node) + '/api/v1/',
                    calls.c.state == 'active',
                )
            )
        )

        active_calls = []
        async for row in self.conn.execute(query):
            active_calls.append(row)

        return active_calls
