from datetime import datetime
import logging
import re
from typing import List, Optional
from urllib.parse import urlparse, parse_qs

from requests.exceptions import RequestException

from django.conf import settings
from django.utils import timezone

from intranet.vconf.src.lib.requests import Session
from intranet.vconf.src.lib.tvm2 import get_service_ticket, get_user_ticket_for_robot

calendar_session = Session()
calendar_session.headers.update({
    'User-agent': 'vconf'
})


logger = logging.getLogger(__name__)


def _get_event_id_by_old_url(url):
    event_url_query = urlparse(url).query
    params = parse_qs(event_url_query)
    event_id = params.get('event_id', []) or params.get('show_event_id', [])

    if event_id:
        event_id = event_id[0]
        if event_id.isdigit():
            return int(event_id)
        else:
            return None
    return None


def _get_event_id_by_new_url(url):
    rgx = re.compile(r'/event/(?P<event_id>\d+)')
    match = rgx.search(url)
    result = match.groupdict() if match else {}
    if 'event_id' in result:
        return int(result['event_id'])
    return


def parse_event_id(raw_event_id):
    """
    Возвращает event_id по URL встречи Календаря

    URL бывает 2х видов:
    1. /event/?event_id=1
    2. /event/1
    """
    if not raw_event_id:
        return
    if isinstance(raw_event_id, int):
        return raw_event_id
    if raw_event_id.isdigit():
        return int(raw_event_id)
    return (
        _get_event_id_by_new_url(raw_event_id)
        or _get_event_id_by_old_url(raw_event_id)
    )


class EventIsMaster(Exception):
    """
    Если текущий event_id является мастер event_id встречи,
    то ручка возвращает ошибку
    """


class CalendarError(Exception):

    def __init__(self, message):
        logger.exception(message)
        super().__init__(message)


class EventDoesNotExist(CalendarError):
    pass


class EventIsNotAvailable(CalendarError):
    pass


def get_from_calendar_api(path, data, user_ticket=None):
    headers = {}

    service_ticket = get_service_ticket(settings.TVM_CALENDAR_CLIENT_ID)
    headers['X-Ya-Service-Ticket'] = service_ticket

    if data.get('uid') is None:
        data['uid'] = settings.VCONF_ROBOT_UID
        user_ticket = get_user_ticket_for_robot()

    assert user_ticket
    headers['X-Ya-User-Ticket'] = user_ticket

    try:
        response = calendar_session.post(
            url=settings.CALENDAR_API_HOST + path,
            data=data,
            headers=headers,
            timeout=(0.5, 1, 5),
        )
    except RequestException:
        raise CalendarError('Calendar API is not responding.')

    try:
        response.raise_for_status()
    except RequestException:
        raise CalendarError('Bad status code from Calendar API: `{}`'.format(response.status_code))

    try:
        result = response.json()
    except ValueError:
        raise CalendarError('Invalid JSON from Calendar API')

    if 'error' in result:
        error_msg = result['error']['message']
        error_code = result['error']['name']

        if error_code == 'event-not-found':
            raise EventDoesNotExist(error_msg)
        elif error_code == 'no-permissions-for-event-action':
            raise EventIsNotAvailable(error_msg)
        elif error_msg == 'expected recurrence event id':
            raise EventIsMaster(error_msg)
        else:
            raise CalendarError(error_msg)

    return result


def get_events_by_time(office_ids: List[str], time: Optional[datetime] = None) -> dict:
    offices = get_events_for_offices(office_ids, time)
    events = dict()
    if not offices:
        return events

    for office in offices:
        for room in office['resources']:
            for event in room['events']:
                if 'eventId' not in event:
                    # если у события нет eventId, значит это резерв и его обрабатывать не нужно
                    continue
                event_id = str(event['eventId'])
                events.setdefault(event_id, {'rooms': []})
                events[event_id]['rooms'].append(room['info']['email'])
                events[event_id]['start_time'] = datetime.strptime(
                    event['start'],
                    '%Y-%m-%dT%H:%M:%S',
                ).replace(tzinfo=timezone.utc)
                events[event_id]['end_time'] = datetime.strptime(
                    event['end'],
                    '%Y-%m-%dT%H:%M:%S',
                ).replace(tzinfo=timezone.utc)

    event_ids = list(events.keys())
    if not event_ids:
        return {}

    events_brief = get_events_brief(
        event_ids=event_ids,
    )
    for event in events_brief['events']:
        event_id = str(event['id'])
        if event.get('othersCanView'):
            events[event_id]['organizer'] = event.get('organizer', {}).get('name', 'Hidden')
            events[event_id]['name'] = event.get('name', 'Meeting name is hidden')
        else:
            events[event_id]['organizer'] = 'Hidden'
            events[event_id]['name'] = 'Meeting name is hidden'

    return events


def get_events_for_offices(office_ids: List[str], time: Optional[datetime] = None) -> list:
    """
    Расписание переговорок по офисам
    https://wiki.yandex-team.ru/calendar/api/new-web/#get-resources-schedule
    """
    if time is None:
        time = datetime.utcnow()
    time = time.strftime('%Y-%m-%dT%H:%M:%S')
    response = get_from_calendar_api(
        path='/get-resources-schedule-by-staff-id',
        data={
            'from': time,
            'to': time,
            'tz': 'UTC',
            'officeId': office_ids,
        },
    )
    return response['offices']


def get_events_brief(event_ids, lang='ru'):
    """
    Краткая информация о своих или чужих событиях
    https://wiki.yandex-team.ru/calendar/api/new-web/#get-events-brief
    """
    return get_from_calendar_api(
        path='/get-events-brief',
        data={
            'eventIds': event_ids,
            'lang': lang,
            'forResource': 'true',
        },
    )


def get_event_info(event_id=None, external_id=None, user=None, master_event=False):
    """
    Информация об одном событии
    https://wiki.yandex-team.ru/calendar/api/new-web/#get-event

    Если передать external_id, то встреча будет отдаваться только в том случае,
    если пользователь как-то связан со встречей (участник или добавлена в календарь).
    """
    assert bool(event_id) != bool(external_id)

    data = {
        'dateFormat': 'zoned',
    }
    if event_id:
        data['eventId'] = event_id
    if external_id:
        data['externalId'] = external_id
    if user:
        data['uid'] = user.uid
    if master_event:
        data['recurrenceAsOccurrence'] = master_event
    return get_from_calendar_api(
        path='/get-event',
        data=data,
        user_ticket=user.user_ticket if user else None,
    )


def get_next_event(event_id=None, user=None):
    """
    Если событие с event_id не является мастер событием, то
    нужно взять master_id из мастер события
    """
    try:
        data = get_event_info(event_id, user=user, master_event=True)
        master_id = data['id']
    except EventIsMaster:
        master_id = event_id

    # по master_id возвращается ближайшее предстоящее событие
    # https://wiki.yandex-team.ru/vconf/dev/eventsync/
    data = get_event_info(master_id, user=user, master_event=False)
    data['master_id'] = master_id
    return data


def get_user_events(user, from_dt=None, to_dt=None):
    """
    События пользователя
    https://wiki.yandex-team.ru/calendar/api/new-web/#get-events

    TODO: Обернуть событие из Календаря в объект
    """
    dt_format = '%Y-%m-%dT%H:%M:%SZ'
    now = timezone.now()
    from_dt = from_dt or now
    to_dt = to_dt or now

    return get_from_calendar_api(
        path='/get-events',
        data={
            'uid': user.uid,
            'from': from_dt.strftime(dt_format),
            'to': to_dt.strftime(dt_format),
            'dateFormat': 'zoned',
            'layerToggledOn': True,
            'hideAbsences': True,
        },
        user_ticket=user.user_ticket,
    )
