import logging
import requests
import random
import defusedxml.ElementTree as ET

from django.conf import settings

from intranet.vconf.src.lib.requests import Session
from intranet.vconf.src.call.models import Node

log = logging.getLogger(__name__)

api_config = settings.CMS_CONFIG

user_jids = settings.CMS_JIDS


class CMSApi:
    class Error(Exception):
        def __init__(self, message):
            log.exception(message)
            super(CMSApi.Error, self).__init__(message)

    def __init__(self, node):
        # Block about CMS API
        self.auth = 'Basic ' + api_config['auth_hash']
        self.host = node
        self.space_filter = api_config['callLegProfiles']['vconf']
        self.streaming_url = api_config['streaming_url']

        # Requests headers
        self.HEADERS = {
            'authorization': self.auth,
            'content-type': "application/x-www-form-urlencoded",
        }
        self.cms_request = Session()
        self.cms_request.headers.update(self.HEADERS)

    @classmethod
    def get_suitable_nodes(cls):
        enable_nodes = Node.objects.filter(enabled=True)
        alive_nodes = []
        for node in enable_nodes:
            if not node.is_alive:
                log.error('Node status is obsolete for node %s', node.id)
                continue
            alive_nodes.append(node)

        alive_nodes.sort(key=lambda x: x.load_available, reverse=True)
        candidates_count = len(alive_nodes) // 2
        if candidates_count:
            return alive_nodes[:candidates_count]
        else:
            log.error('There is not suitable nodes. Have to take all.')
            return enable_nodes

    @classmethod
    def get_node(cls):
        nodes = cls.get_suitable_nodes()
        log.info('Choosing from nodes (%s)', ', '.join(str(node.id) for node in nodes))
        return random.choice(nodes)

    @classmethod
    def xml_to_dict(cls, xmltext):
        tree = ET.fromstring(xmltext)
        result = []
        for child in tree:
            d = child.attrib
            for param in child:
                d.update({param.tag: param.text})
            result.append(d)
        return result

    @staticmethod
    def get_free_owner() -> str:
        return random.choice(user_jids)

    def space_create(self, call_id, uri, name, secret, stream_id, owner, stream=False, record=False):
        payload = {
            'name': name,
            'uri': uri,
            'defaultLayout': 'automatic',
            'callId': call_id,
            'secret': secret,
            'callLegProfile': self.space_filter,
            'streamURL': f'{self.streaming_url}/{stream_id}/{stream_id}0',
            'ownerJid': owner,
            'callProfile': {
                (True, True): api_config['callProfiles']['liverec'],
                (True, False): api_config['callProfiles']['live'],
                (False, True): api_config['callProfiles']['rec'],
                (False, False): '',
            }[(stream, record)]
        }

        url = self.host + 'coSpaces'
        response = requests.request('POST', url, data=payload, headers=self.HEADERS, verify=False)
        log.debug('Space creation response: %s %s', response.status_code, response.text)
        return response

    def space_delete(self, conf_id):
        url = self.host + 'coSpaces/' + conf_id
        response = requests.request('DELETE', url, headers=self.HEADERS, verify=False)
        return response

    def call_create(self, conf_id, call_name):
        payload = {
            'coSpace': conf_id,
            'name': call_name,
        }
        url = self.host + 'calls'
        response = requests.request('POST', url, data=payload, headers=self.HEADERS, verify=False)
        log.debug('Call creation response: %s %s', response.status_code, response.text)
        return response

    def participant_add(self, uri, call_id, label):
        url = self.host + 'calls/' + call_id + '/participants'
        payload = {'remoteParty': uri, 'nameLabelOverride': label}
        response = requests.request('POST', url, data=payload, headers=self.HEADERS, verify=False)
        return response

    def participant_delete(self, pt_id):
        url = self.host + 'participants/' + pt_id
        response = requests.request('DELETE', url, headers=self.HEADERS, verify=False)
        return response

    def get_calllegs(self, pt_id):
        search_url = self.host + 'callLegs'
        params = {
            'participantFilter': pt_id,
        }

        response = requests.request('GET', search_url, params=params, headers=self.HEADERS, verify=False)
        log.debug('Call legs response: %s %s', response.status_code, response.text)

        return response

    def participant_mute(self, pt_id, actions):
        response = self.get_calllegs(pt_id)

        calllegs = self.xml_to_dict(response.text)
        if calllegs:
            for callleg in calllegs:
                url = self.host + 'callLegs/' + callleg['id']
                payload = {
                    'rxAudioMute': not actions['mic'],
                    'rxVideoMute': not actions['cam'],
                }

                response = requests.request('PUT', url, data=payload, headers=self.HEADERS, verify=False)
                log.debug('Mute toggle response: %s %s', response.status_code, response.text)
                return response
        else:
            log.info('Call legs for participant %s was not found', pt_id)
            return response

    def system_load(self):
        url = self.host + 'system/load'
        response = self.cms_request.get(
            url=url,
            timeout=(0.5, 1, 2),
            verify=False,
        )
        tree = ET.fromstring(response.text)
        element = tree.find('mediaProcessingLoad')
        result = int(element.text)
        return result
