import logging
from requests import Session
from requests.auth import HTTPBasicAuth

from django.conf import settings

from zeep import Client
from zeep.cache import SqliteCache
from zeep.transports import Transport
from zeep.plugins import HistoryPlugin

log = logging.getLogger(__name__)


class _SessionWrapper(Session):
    """
    Сертификата /etc/allCAs.pem нет в /etc/ssl/certs/, а его явная передача
    не помогает из-за бага в методе merge_environment_settings (requests==2.27.1)
    При этом zeep использует этот метод. Чтобы сертификат дошел, явно передаем в методе
    именно его
    """
    def merge_environment_settings(self, url, proxies, stream, verify, cert):
        return super().merge_environment_settings(url, proxies, stream, '/etc/allCAs.pem', cert)


class CUCMClient:
    """
    CUCM – Cisco Unified Communications Manager

    Отсюда можно вытащить информацию об имеющихся устройствах
    Пока известно только два типа подключения (service_type)
    - AXL – позволяет вытащить информацию о наличии устройств
    - RIS – позволяет получить подробную информацию об устройствах

    Работает всё это по протоколу soap, для работы с ним используется
    библиотека zeep

    Подробнее:
    - Дока от Cisco: https://www.ciscopress.com/articles/article.asp?p=1216890&seqNum=2
    - CUCM в Яндексе: https://wiki.yandex-team.ru/noc/office/sucm/
    """

    class Error(Exception):
        def __init__(self, message):
            log.exception(message)
            super().__init__(message)

    def __init__(self, service_type: str):
        if service_type not in settings.CUCM_SERVICES_CONFIG:
            raise self.Error(f'Unknown service type {service_type}')

        self.bind = settings.CUCM_SERVICES_CONFIG[service_type]['bind']
        self.url = settings.CUCM_SERVICES_CONFIG[service_type]['url']
        self.wsdl = settings.CUCM_SERVICES_CONFIG[service_type]['wsdl']

        self.session = _SessionWrapper()
        self.session.auth = HTTPBasicAuth(settings.CUCM_USERNAME, settings.CUCM_PASSWORD)

        self.transport = Transport(cache=SqliteCache(), session=self.session, timeout=20)
        self.history = HistoryPlugin()
        self.soap_client = Client(wsdl=self.wsdl, transport=self.transport, plugins=[self.history])

        self.service = self.soap_client.create_service(self.bind, self.url)

    def get_codec_number_to_ipv6_and_model_map(self, numbers) -> dict[str, dict]:
        numbers = [int(num) for num in numbers if num]
        fltr = {
            'MaxReturnedDevices': '1000',
            'DeviceClass': 'Phone',
            'Model': '255',
            'Status': 'Any',
            'NodeName': '',
            'SelectBy': 'DirNumber',
            'SelectItems': {
                'item': numbers,
            },
            'Protocol': 'Any',
            'DownloadStatus': 'Any',
        }

        response = self.service.selectCmDeviceExt(CmSelectionCriteria=fltr, StateInfo='')

        devices = []
        number_to_cucm_data_map = {}

        for node in response['SelectCmDeviceResult']['CmNodes']['item']:
            if node['ReturnCode'] == 'Ok':
                devices.extend(node['CmDevices']['item'])

        for device in devices:
            try:
                number = device['LinesStatus']['item'][0]['DirectoryNumber']
                ipv6 = shrink_ipv6(device['IPAddress']['item'][1]['IP'])
                number_to_cucm_data_map[number] = {
                    'ip': f'[{ipv6}]',
                    'model': device['Model'],
                }
            except (KeyError, IndexError):
                log.exception('Error while getting IPv6 for room %s', device['DirNumber'])

        return number_to_cucm_data_map


def shrink_ipv6(ipv6: str):
    """
    Убирает лишние нули в IPv6
    >>> shrink_ipv6('2a02:06b8:0000:0423:42ce:24ff:fe0e:d1b9')
    '2a02:6b8:0:423:42ce:24ff:fe0e:d1b9'
    """
    return ':'.join(hex(int(i, 16))[2:] for i in ipv6.split(':'))
