from uuid import uuid4
import logging

import requests
from django.conf import settings
from django.core.cache import cache

from intranet.vconf.src.lib.requests import Session
from intranet.vconf.src.lib.tvm2 import get_service_ticket


logger = logging.getLogger(__name__)


class MessengerAPIError:
    pass


class MessengerAPI:

    session = Session()
    session.headers['User-agent'] = 'vconf'

    def _request(self, method, params):
        self.session.headers['X-Ya-Service-Ticket'] = get_service_ticket(settings.TVM_MESSENGER_CLIENT_ID)
        request_id = str(uuid4())
        self.session.headers['X-REQUEST-ID'] = request_id
        logger.info('X-REQUEST-ID: %s', request_id)

        try:
            response = self.session.post(
                url=settings.MESSENGER_API_URL,
                json={
                    'method': method,
                    'params': params,
                },
                timeout=settings.MESSENGER_API_TIMEOUT,
            )
        except requests.RequestException:
            logger.exception('Failed request to Messenger API, method: %s', method)
            raise MessengerAPIError

        try:
            data = response.json()
        except ValueError:
            logger.exception('Error during parse json from Messenger API')
            raise MessengerAPIError

        if data['status'] == 'error':
            logger.error(
                'Messenger API responded with status code `%d`: %s',
                response.status_code,
                data['data']['text'],
            )
            raise MessengerAPIError

        return data['data']

    def _get_guid(self, login):
        key = 'messenger_guid_by_login_' + login
        guid = cache.get(key)
        if guid:
            return guid

        data = self._request(
            method='request_user',
            params={
                'nickname': login,
            },
        )
        guid = data['user']['guid']
        cache.set(key, guid, timeout=60 * 60)  # hour
        return guid

    def create_chat(self, name, description, admin_logins, member_logins, chat_id=None, is_hural=False):
        admin_guids = [self._get_guid(login) for login in admin_logins]
        member_guids = [self._get_guid(login) for login in member_logins]

        params = {
            'name': name,
            'description': description,
            'roles': {
                'admin': admin_guids,
            },
            'members': member_guids,
            'public': True,
            'org_id': settings.YA_TEAM_ORG_ID,
        }

        if is_hural:
            params['avatar_id'] = settings.MESSENGER_HURAL_AVATAR_ID
            params['key'] = settings.MESSENGER_HURAL_AVATAR_KEY

        if chat_id:
            params['chat_id'] = chat_id

        data = self._request(
            method='create_chat',
            params=params,
        )
        return data

    def stop_chat(self, chat_id):
        data = self._request(
            method='update_chat',
            params={
                'chat_id': chat_id,
                'metadata': {
                    'member_rights': {
                        'write': False,
                    },
                },
            },
        )
        return data


MessengerAPI = MessengerAPI()
