import json
import logging
import os
import socket
import time
import traceback

import yenv

from django.conf import settings


class ErrorBoosterHandler(logging.Handler):

    @staticmethod
    def _get_traceback(record: logging.LogRecord) -> str:
        _tb = None
        if record.exc_info:
            _, _, _tb = record.exc_info
        tb = ''.join(traceback.extract_tb(_tb).format()) if _tb else ''

        return tb

    def emit(self, record: logging.LogRecord) -> None:

        if record.levelno < logging.ERROR:
            return

        # Список полей: https://wiki.yandex-team.ru/error-booster/error-how-to/#kakispolzovat
        message = {
            # обязательные поля
            'project': settings.ERROR_BOOSTER_PROJECT,
            'timestamp': round(time.time() * 1000),
            'message': record.msg,
            # необязательные поля
            'env': yenv.type,
            'file': record.name,
            'level': record.levelname,
            'source': os.getenv('DEPLOY_UNIT_ID'),
            'dc': os.getenv('DEPLOY_NODE_DC'),
            'stack': self._get_traceback(record),
            'line': record.lineno,
            'additional': {
                'formatMessage': record.getMessage(),
            },
        }

        sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        sock.sendto(
            bytes(json.dumps(message), 'utf-8'),
            (settings.ERROR_BOOSTER_HOST, settings.ERROR_BOOSTER_PORT),
        )
