import types
from json import JSONEncoder, dumps
from datetime import date, time, datetime
from decimal import Decimal
from django.http import HttpResponse
from functools import wraps

from intranet.vconf.src.lib.exceptions import HttpError


class CustomJSONEncoder(JSONEncoder):
    def default(self, o):
        if isinstance(o, datetime):
            return o.isoformat()
        if isinstance(o, (time, date)):
            return o.isoformat()
        elif isinstance(o, set):
            return list(o)
        elif isinstance(o, Decimal):
            return str(o)
        elif isinstance(o, types.GeneratorType):
            return list(o)
        else:
            return super(CustomJSONEncoder, self).default(o)


def make_json_response(data, status=200, data_field_name='response_text'):
    data = dumps(
        {'response_code': status, data_field_name: data},
        cls=CustomJSONEncoder
    )
    return HttpResponse(
        data,
        content_type='application/json; charset=utf-8',
        status=status,
    )


def make_error_response(error):
    return make_json_response(
        data={
            'code': error.code,
            'message': error.message,
            'details': error.details,
        },
        status=error.status,
    )


def responding_json(view):
    """
    :param view: Декорируемая view. Может возвращать все что превращается в json
    :return: HttpResponse где правильный content_type и данные это json
    """
    @wraps(view)
    def wrapper(request, *args, **kwargs):
        try:
            data = view(request, *args, **kwargs)
        except HttpError as e:
            return make_error_response(e)

        if isinstance(data, HttpResponse):
            return data

        status = 200
        if isinstance(data, tuple):
            data, status = data

        return make_json_response(data, status)

    return wrapper
