import re
import logging

try:
    import uwsgi
except ImportError:
    uwsgi = None

from django.contrib.auth.models import AnonymousUser
from django.contrib.auth import get_user_model
from django.conf import settings
from django.core.exceptions import PermissionDenied
from django.utils.deprecation import MiddlewareMixin
from django_yauth.middleware import YandexAuthMiddleware
from django_yauth.exceptions import AuthException
from django_yauth.util import get_real_ip
from django_yauth.user import create_user, YandexUser, AnonymousYandexUser

from intranet.vconf.src.lib.json import make_json_response

logger = logging.getLogger(__name__)
User = get_user_model()


class AuthMiddleware(YandexAuthMiddleware):

    def process_request(self, request):
        if request.path == '/cdr/':
            token = request.GET.get('token')

            if not token or token != settings.CDR_TOKEN:
                return make_json_response('unauthorized', status=401)
            ip = get_real_ip(request)
            if ip not in settings.CMS_IPS:
                logger.warning('Not valid ip %s in cdr/', ip)
                return make_json_response('unauthorized', status=401)

            return

        if request.path.startswith('/codec/webhook'):
            token = request.GET.get('token')
            if not token or token != settings.CODEC_WEBHOOK_TOKEN:
                return make_json_response('unauthorized', status=401)
            return

        self.assign_yauser(request)
        try:
            if request.yauser is None or not request.yauser.is_authenticated():
                for path in getattr(settings, 'YAUTH_IGNORE_PATHS', []):
                    rgx = re.compile(path)
                    if rgx.match(request.path):
                        break
                else:
                    return make_json_response('unauthorized', status=401)
        except AuthException:
            return make_json_response('unauthorized', status=401)

        # IDM
        if request.path.startswith('/' + settings.IDM_URL_PREFIX):
            if (not request.yauser.service_ticket
                    or request.yauser.service_ticket.src != settings.IDM_TVM_CLIENT_ID):
                raise PermissionDenied('TVM auth failed')

        self.assign_user(request)
        if uwsgi:
            self.uwsgi_log(request)

    def assign_user(self, request):
        yauser = request.yauser
        if yauser and yauser.is_authenticated() and 'login' in yauser.fields:
            try:
                user = User.objects.get(username=yauser.login)
            except User.DoesNotExist:
                user = create_user(yauser)
            user._extra = {
                'uid': yauser.uid,
                'lang': yauser.language,
                'ip': get_real_ip(request),
                'secret': request.GET.get('secret'),
                'user_ticket': yauser.raw_user_ticket,
            }
            request.user = user
        else:
            request.user = AnonymousUser()

    def uwsgi_log(self, request):
        try:
            if request.yauser:
                uwsgi.set_logvar('yandex_login', request.yauser.login)

                if request.yauser.oauth:
                    uwsgi.set_logvar(
                        'client_application',
                        request.yauser.oauth.client_name.encode('utf-8'),
                    )
                    uwsgi.set_logvar('client_id', request.yauser.oauth.client_id)
        except Exception:
            logger.exception('Exception on set uWSGI logvar')


class AuthTestMiddleware(MiddlewareMixin):
    """
    Аутентификация для использования в тестах
    """
    def process_request(self, request):
        username = getattr(settings, 'AUTH_TEST_USER', None)
        email = username and username + '@yandex-team.ru'
        mechanism_name = 'cookie'

        class FakeMechanism(object):
            def __init__(self):
                self.mechanism_name = mechanism_name

        try:
            user = User.objects.get(username=username)
        except User.DoesNotExist:
            request.yauser = AnonymousYandexUser()
            request.user = AnonymousUser()
            return

        uid = '000000'
        request.yauser = YandexUser(
            uid=uid,
            fields={},
            need_reset=False,
            emails=[email],
            default_email=email,
            mechanism=FakeMechanism(),
            raw_user_ticket='123',
        )
        user._extra = {
            'uid': uid,
            'is_external': False,
            'is_ip_external': False,
            'affiliation': 'yandex',
            'is_admin': True,
            'tz': 'Europe/Moscow',
            'lang': 'ru',
            'raw_user_ticket': '123',
        }
        request.user = user
