from requests import sessions
from requests.models import parse_url
import urllib3

from requests.exceptions import RequestException

from django.conf import settings

import logging


urllib3.disable_warnings()


class HostDisabledException(Warning):
    pass


class Session(sessions.Session):
    logger_prefix = 'requests'

    @staticmethod
    def check_disabled(host):
        if host in settings.DISABLED_HOSTS:
            raise HostDisabledException('request to {host} has been cancelled'.format(host=host))

    @staticmethod
    def get_context(method, parsed_url, **kwargs):
        context = kwargs.copy()
        context['method'] = method.upper()
        context.update(parsed_url._asdict())
        context.pop('files', None)
        context.pop('headers', None)
        if 'data' in context and not isinstance(context['data'], (str, dict)):
            context.pop('data', None)

        return context

    def request(self, method, url, **kwargs):
        """
            log_message - текст, попадающий в лог.
            timeout - можно задать списком, тогда будут делаться ретраи.
            read_timeout - Если задан, то первый таймаут - на коннекшн, а этот - на чтение данных. Строго число.
        """

        parsed_url = parse_url(url)
        self.check_disabled(parsed_url.host)
        logger = logging.getLogger('{prefix}__{host}'.format(prefix=self.logger_prefix, host=parsed_url.host))

        timeouts = kwargs.pop('timeout', None)
        if not timeouts:
            logger.warning('No timeout set while requesting %s', url)
            timeouts = 2

        read_timeout = kwargs.pop('read_timeout', None)

        if isinstance(timeouts, (int, float)):
            timeouts = (timeouts, )

        log_message = str(kwargs.pop('log_message', ''))

        for attempt_no, timeout in enumerate(timeouts, start=1):
            actual_timeout = timeout if not read_timeout else (timeout, read_timeout)
            try:
                response = super(Session, self).request(method, url, timeout=actual_timeout, **kwargs)
            except RequestException as e:
                logger.debug('%s request to `%s` raises: %s', method, url, e)
                if attempt_no == len(timeouts):
                    raise
            else:
                log_msg = ''
                if attempt_no > 1 or log_message:
                    log_msg = '(attempt: {no}; {msg})'.format(no=attempt_no, msg=log_message)

                elapsed = int(response.elapsed.total_seconds() * 1000)
                # request_context = self.get_context(
                #     method,
                #     parsed_url,
                #     elapsed=elapsed,
                #     status_code=response.status_code,
                #     **kwargs
                # )
                # with log_context(**request_context):
                logger.info(
                    '%s>%s request to `%s` %s elapsed in %s ms.',
                    method.upper(), response.status_code, url, log_msg, elapsed,
                )
                return response
